package dev.fitko.fitconnect.api.services.submission;

import dev.fitko.fitconnect.api.domain.limits.submission.SubmissionLimits;
import dev.fitko.fitconnect.api.domain.model.destination.Destination;
import dev.fitko.fitconnect.api.domain.model.destination.PublicDestination;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.ApiAttachment;
import dev.fitko.fitconnect.api.domain.model.metadata.data.Data;
import dev.fitko.fitconnect.api.domain.model.submission.AnnounceSubmission;
import dev.fitko.fitconnect.api.domain.model.submission.CreatedSubmission;
import dev.fitko.fitconnect.api.domain.model.submission.PublicService;
import dev.fitko.fitconnect.api.domain.model.submission.Submission;
import dev.fitko.fitconnect.api.domain.model.submission.SubmissionsForPickup;
import dev.fitko.fitconnect.api.domain.model.submission.SubmitSubmission;
import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import java.io.InputStream;
import java.util.UUID;

/**
 * A service that provides access to the FIT-Connect REST-API and allows creating and loading {@link
 * Submission}s and {@link ApiAttachment}s.
 *
 * @see <a href="https://docs.fitko.de/fit-connect/docs/apis/submission-api">FIT-Connect Submission
 *     API</a>
 */
public interface SubmissionService {

    /**
     * Announce a new submission. This step is necessary before actually {@link
     * #sendSubmission(SubmitSubmission) sending} the submission.
     *
     * @param submission submission that contains all {@link ApiAttachment}s that should be announced,
     *     as well as a specified {@link PublicService}
     * @return announced submission with submissionId, caseId, and destinationID
     * @throws RestApiException if the submission couldn't be announced or a technical error occurred
     * @see <a href="https://docs.fitko.de/fit-connect/docs/sending/start-submission">Announcing a
     *     submission</a>
     */
    CreatedSubmission announceSubmission(AnnounceSubmission submission) throws RestApiException;

    /**
     * Send a submission that was already {@link #announceSubmission(AnnounceSubmission) announced}
     * before.
     *
     * @param submission submission including the encrypted {@link Data} and {@link Metadata}
     * @return the submission that was sent
     * @throws RestApiException if the submission couldn't be sent or a technical error occurred
     */
    Submission sendSubmission(SubmitSubmission submission) throws RestApiException;

    /**
     * Get a {@link Submission} by id.
     *
     * @param submissionId unique submission identifier
     * @return submission matching the given id
     * @throws RestApiException if the submission is not available or a technical error occurred
     */
    Submission getSubmission(UUID submissionId) throws RestApiException;

    /**
     * Get all available submissions for a given {@link Destination}.
     *
     * @param destinationId unique destination identifier
     * @param offset position in the dataset
     * @param limit number of submissions in result (max. is 500)
     * @return SubmissionsForPickup containing a list of submissions
     * @throws RestApiException if the submissions couldn't be polled or a technical error occurred
     */
    SubmissionsForPickup pollAvailableSubmissionsForDestination(UUID destinationId, int offset, int limit)
            throws RestApiException;

    /**
     * Upload an encrypted {@link ApiAttachment}.
     *
     * @param submissionId unique identifier of an announced submission
     * @param attachmentId unique destination identifier
     * @param encryptedAttachment JWE encrypted attachment payload
     * @throws RestApiException if a technical error occurred
     */
    void uploadAttachment(UUID submissionId, UUID attachmentId, String encryptedAttachment) throws RestApiException;

    /**
     * Upload an encrypted {@link ApiAttachment} with input-stream payload.
     *
     * @param submissionId unique identifier of an announced submission
     * @param attachmentId unique destination identifier
     * @param encryptedAttachment JWE encrypted attachment payload as input-stream
     * @throws RestApiException if a technical error occurred
     */
    void uploadAttachmentStream(UUID submissionId, UUID attachmentId, InputStream encryptedAttachment)
            throws RestApiException;

    /**
     * Get an {@link ApiAttachment} by id for a given {@link Submission}.
     *
     * @param submissionId unique submission identifier
     * @param attachmentId unique attachment identifier
     * @return encrypted string of the attachment data
     * @throws RestApiException if the attachment is not available or a technical error occurred
     */
    String getAttachment(UUID submissionId, UUID attachmentId) throws RestApiException;

    /**
     * Get attachment limits that apply to sending submissions to this destination. <br>
     * <i> This endpoint returns different results depending on whether the request is made with a
     * SENDER or SUBSCRIBER client. The endpoint can be used to find out which limitations apply to a
     * specific client. </i>
     *
     * @param destinationId unique identifier of the destination
     * @return the {@link SubmissionLimits} for the requested destination
     * @throws RestApiException if the limits aren't available or a technical error occurred
     */
    SubmissionLimits getDestinationAttachmentLimits(UUID destinationId) throws RestApiException;

    /**
     * Get attachment limits that apply to sending submissions and replies related to a case. <br>
     * <i> This endpoint returns different results depending on whether the request is made with a
     * SENDER or SUBSCRIBER client. The endpoint can be used to find out which limitations apply to a
     * specific client. </i>
     *
     * @param caseId unique identifier of the case
     * @return the {@link SubmissionLimits} for the requested case
     * @throws RestApiException if the limits aren't available or a technical error occurred
     */
    SubmissionLimits getCaseAttachmentLimits(UUID caseId) throws RestApiException;

    /**
     * Get a {@link PublicDestination} by id.
     * This destination is fetched from the SubmissionAPI.
     *
     * @param destinationId unique destination identifier
     * @return the {@link Destination}
     * @throws RestApiException if the destination is not available or error occurred
     */
    PublicDestination getDestination(UUID destinationId) throws RestApiException;
}
