package dev.fitko.fitconnect.api.services.schema;

import dev.fitko.fitconnect.api.exceptions.internal.SchemaNotFoundException;
import java.net.URI;

/**
 * Central schema provider that gives access to the current schemas for metadata and the
 * security-event-token.
 *
 * @see <a href="https://docs.fitko.de/fit-connect/docs/metadata/overview">Metadata schema</a>, <a
 *     href="https://docs.fitko.de/fit-connect/docs/set-schema/overview">Security-Event-Token
 *     Schema</a>
 */
public interface SchemaProvider {

    /**
     * Test if a given set-schema uri is an allowed configuration.
     *
     * @param schemaUri uri of the schema to test
     * @return true if the schema is provided, false if not
     */
    boolean isAllowedSetSchema(URI schemaUri);

    /**
     * Test if a given metadata-schema uri is an allowed configuration.
     *
     * @param schemaUri uri of the schema to test
     * @return true if the schema is provided, false if not
     */
    boolean isAllowedMetadataSchema(URI schemaUri);

    /**
     * Load the set-schema for a given schema uri.
     *
     * @return the schema as string
     * @throws SchemaNotFoundException if the schema is not existing
     */
    String loadSetSchema(URI schemaUri) throws SchemaNotFoundException;

    /**
     * Load the metadata-schema for a given schema uri.
     *
     * @param schemaUri uri of the metadata-schema that should be loaded
     * @return the schema as string
     * @throws SchemaNotFoundException if the schema is not existing
     */
    String loadMetadataSchema(URI schemaUri) throws SchemaNotFoundException;

    /**
     * Load the schema for payload of signed destination.
     *
     * @param schemaUri uri of the destination schema that should be loaded
     * @return the schema as string
     * @throws SchemaNotFoundException if the schema is not existing
     */
    String loadDestinationSchema(URI schemaUri) throws SchemaNotFoundException;

    /**
     * Load the schema for the submission data.
     *
     * @param schemaUri uri of the submission data schema that should be loaded
     * @return the schema as string
     * @throws SchemaNotFoundException if the schema is not existing
     */
    String loadSubmissionDataSchema(final URI schemaUri) throws SchemaNotFoundException;
}
