package dev.fitko.fitconnect.api.services.keys;

import com.nimbusds.jose.jwk.KeyOperation;
import com.nimbusds.jose.jwk.RSAKey;
import dev.fitko.fitconnect.api.domain.model.destination.Destination;
import dev.fitko.fitconnect.api.domain.model.destination.PublicDestination;
import dev.fitko.fitconnect.api.services.validation.ValidationService;
import java.util.Date;
import java.util.UUID;

/**
 * Service to retrieve and validate public keys from a {@link PublicDestination} or the submission service
 * well-known keys.
 *
 * @see ValidationService
 */
public interface KeyService {

    /**
     * Get the public encryption key for a given destination.
     *
     * @param destination the {@link PublicDestination} the key is requested from
     * @return validated {@link RSAKey} (@see {@link ValidationService#validatePublicKey(RSAKey,
     *     KeyOperation)}})
     */
    RSAKey getPublicEncryptionKey(PublicDestination destination);

    /**
     * Get the public signature key for a given destination.
     *
     * @param destinationId unique identifier of the {@link Destination}
     * @param keyId unique identifier of the {@link RSAKey}
     * @param validationDate date for which the validity of the certificate chain (x5c) should be
     *     determined
     * @return validated {@link RSAKey} (@see {@link ValidationService#validatePublicKey(RSAKey,
     *     KeyOperation)}})
     */
    RSAKey getDestinationPublicSignatureKey(UUID destinationId, String keyId, Date validationDate);

    /**
     * Get a public signature key for a given key-id from the baseUrl well-known keys.
     *
     * @param keyId unique identifier of the {@link RSAKey}
     * @param baseUrl the base url the key is fetched from
     * @param validationDate date for which the validity of the certificate chain (x5c) should be
     *     determined
     * @return validated {@link RSAKey} (@see {@link ValidationService#validatePublicKey(RSAKey,
     *     KeyOperation)}})
     */
    RSAKey getPublicSignatureWellKnownKey(String keyId, String baseUrl, Date validationDate);

    /**
     * Get a public signature key for a given key-id from the submission service well-known keys.
     *
     * @param keyId unique identifier of the {@link RSAKey}
     * @param validationDate date for which the validity of the certificate chain (x5c) should be
     *     determined
     * @return validated {@link RSAKey} (@see {@link ValidationService#validatePublicKey(RSAKey,
     *     KeyOperation)}})
     */
    RSAKey getSubmissionServicePublicSignatureKey(String keyId, Date validationDate);
}
