package dev.fitko.fitconnect.api.services.http;

import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import java.util.Map;

/**
 * HTTP-Service that provides basic HTTP verbs for GET, PUT, POST, PATCH, and DELETE requests. This
 * service is used internally by the SDK to communicate with the FIT-Connect REST API. It handles
 * all HTTP communication, including - Request/response handling - Header management - Error
 * handling - Response type conversion
 */
public interface HttpClient {

    /**
     * Send HTTP GET request to a given request url.
     *
     * @param url the URL the request is sent to
     * @param headers map with HTTP headers
     * @param responseType the response type of the requested payload
     * @return {@link HttpResponse}
     * @throws RestApiException if an error occurred during the request
     */
    <R> HttpResponse<R> get(final String url, final Map<String, String> headers, final Class<R> responseType)
            throws RestApiException;

    /**
     * Send HTTP PUT request with payload to a given request url.
     *
     * @param url the URL the request is sent to
     * @param headers map with HTTP headers
     * @param httpPayload the payload that is sent
     * @param responseType the response type of the request
     * @param <P> type of the payload
     * @param <R> type of the requests' response
     * @return {@link HttpResponse}
     * @throws RestApiException if an error occurred during the request
     */
    <P, R> HttpResponse<R> put(
            final String url, final Map<String, String> headers, final P httpPayload, final Class<R> responseType)
            throws RestApiException;

    /**
     * Send HTTP POST request with payload to a given request url.
     *
     * @param url the URL the request is sent to
     * @param headers map with HTTP headers
     * @param httpPayload the payload that is sent
     * @param responseType the response type of the request
     * @param <P> type of the payload
     * @param <R> type of the requests' response
     * @return {@link HttpResponse}
     * @throws RestApiException if an error occurred during the request
     */
    <P, R> HttpResponse<R> post(
            final String url, final Map<String, String> headers, final P httpPayload, final Class<R> responseType)
            throws RestApiException;

    /**
     * Send HTTP PATCH request with payload to a given request url.
     *
     * @param url the URL the request is sent to
     * @param headers map with HTTP headers
     * @param httpPayload the payload that is sent
     * @param responseType the response type of the request
     * @param <P> type of the payload
     * @param <R> type of the requests' response
     * @return {@link HttpResponse}
     * @throws RestApiException if an error occurred during the request
     */
    <P, R> HttpResponse<R> patch(
            final String url, final Map<String, String> headers, final P httpPayload, final Class<R> responseType)
            throws RestApiException;

    /**
     * Send HTTP DELETE request to a given request url.
     *
     * @param url the URL the request is sent to
     * @param headers map with HTTP headers
     * @return {@link HttpResponse}
     * @throws RestApiException if an error occurred during the request
     */
    HttpResponse<Void> delete(String url, Map<String, String> headers) throws RestApiException;
}
