package dev.fitko.fitconnect.api.services.events;

import dev.fitko.fitconnect.api.domain.model.cases.Case;
import dev.fitko.fitconnect.api.domain.model.cases.Cases;
import dev.fitko.fitconnect.api.domain.model.event.EventLog;
import dev.fitko.fitconnect.api.domain.model.event.EventLogEntry;
import dev.fitko.fitconnect.api.domain.model.event.Status;
import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.authtags.ValidatedAuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.reply.Reply;
import dev.fitko.fitconnect.api.domain.model.reply.SentReply;
import dev.fitko.fitconnect.api.domain.model.submission.SentSubmission;
import dev.fitko.fitconnect.api.domain.model.submission.Submission;
import dev.fitko.fitconnect.api.exceptions.internal.EventLogException;
import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import java.util.List;
import java.util.UUID;

/**
 * A service that provides access to the {@link Submission}s {@link EventLog} and {@link Case} via
 * the FIT-Connect REST-API. Every event is a recorded step within the process of sending a
 * submission or a case. Events are encoded as security event tokens (SET). *
 *
 * @see <a href="https://docs.fitko.de/fit-connect/docs/getting-started/event-log/events">Event
 *     Documentation</a>
 * @see <a href="https://docs.fitko.de/fit-connect/docs/apis/submission-api">Submission-API</a>
 */
public interface CaseService {

    /**
     * Get the {@link EventLog} for a case.
     *
     * @param destinationId unique identifier of the destination
     * @param caseId unique case identifier
     * @param authenticationTags
     * @return list of {@link EventLogEntry}
     * @throws EventLogException if a technical error occurred or the validation failed
     */
    List<EventLogEntry> getEventLogForCase(UUID destinationId, UUID caseId, AuthenticationTags authenticationTags);

    /**
     * Get the {@link EventLog} for a submission.
     *
     * @param destinationId unique identifier of the destination
     * @param caseId unique case identifier
     * @param authenticationTags
     * @return list of {@link EventLogEntry}
     * @throws EventLogException if a technical error occurred or the validation failed
     */
    List<EventLogEntry> getEventLogForSubmission(
            UUID destinationId, UUID caseId, UUID submissionId, AuthenticationTags authenticationTags);

    /**
     * Lists all {@link Cases} that are visible to the client.
     *
     * @param limit max entries per request
     * @param offset skip n elements
     * @return list of cases
     * @throws RestApiException if a technical error occurred
     */
    Cases listCases(int limit, int offset) throws RestApiException;

    /**
     * Load a specific {@link Case} for a caseId.
     *
     * @return {@link Case}
     * @throws RestApiException if a technical error occurred
     */
    Case getCase(UUID caseId) throws RestApiException;

    /**
     * Retrieve the authentication tags for the submitted state of a submission.
     *
     * @return {@link ValidatedAuthenticationTags} of the submit event
     * @throws EventLogException if a validation error occurred
     * @throws RestApiException if a technical error occurred
     */
    ValidatedAuthenticationTags getAuthenticationTags(Submission submission) throws RestApiException, EventLogException;

    /**
     * Retrieve the authentication tags for the submitted state of a reply.
     *
     * @return {@link ValidatedAuthenticationTags} of the submit event
     * @throws EventLogException if a validation error occurred
     * @throws RestApiException if a technical error occurred
     */
    ValidatedAuthenticationTags getAuthenticationTags(Reply reply) throws RestApiException, EventLogException;

    /**
     * Send an event for a given caseId.
     *
     * @param caseId unique case identifier
     * @param signedAndSerializedSET the serialised and signed event as SET string
     * @throws EventLogException if a validation error occurred
     * @throws RestApiException if a technical error occurred
     */
    void sendEvent(UUID caseId, String signedAndSerializedSET) throws RestApiException, EventLogException;

    /**
     * Retrieve the latest status of a {@link Submission}.
     *
     * @return {@link Status} the current status
     * @throws EventLogException if a validation failed
     * @throws RestApiException if a technical error occurred
     */
    Status getStatus(SentSubmission sentSubmission) throws RestApiException, EventLogException;

    /**
     * Retrieve the latest status of a {@link Reply}.
     *
     * @return {@link Status} the current status
     * @throws EventLogException if a validation error occurred
     * @throws RestApiException if a technical error occurred
     */
    Status getStatus(SentReply reply) throws RestApiException, EventLogException;

    /**
     * Retrieve submitted state of a submission.
     *
     * @return {@link Status} the submitted status
     * @throws EventLogException if a validation error occurred
     * @throws RestApiException if a technical error occurred
     */
    Status getSubmissionSubmitState(UUID caseId, UUID submissionId) throws RestApiException, EventLogException;
}
