package dev.fitko.fitconnect.api.services.crypto;

import java.io.InputStream;

/** Provides the creation and verification of hashes. */
public interface MessageDigestService {

    /**
     * Create a hash-sum of a given byte[].
     *
     * @param data input bytes that will be hashed
     * @return byte[] of the hash
     */
    byte[] createHash(byte[] data);

    /**
     * Create a hash-sum of a given input stream.
     *
     * @param inputStream input stream data that will be hashed
     * @return byte[] of the hash
     */
    byte[] createHash(InputStream inputStream);

    /**
     * Create a new hash of a given byte[] and compare it with an existing hash on equality.
     *
     * @param originalHash original hash as byte[]
     * @param data data as byte[] that will be hashed
     * @return boolean if the two hashes are equal
     */
    boolean verify(byte[] originalHash, byte[] data);

    /**
     * Create a new hash of a given input-stream and compare it with an existing hash on equality.
     *
     * @param originalHash original hash as byte[]
     * @param data data as input-stream that will be hashed
     * @return boolean if the two hashes are equal
     */
    boolean verify(byte[] originalHash, InputStream data);

    /**
     * Convert hash to a hexadecimal string representation.
     *
     * @param hash input hash as byte[]
     * @return hex string of the given hash
     */
    String toHexString(byte[] hash);

    /**
     * Convert a hexadecimal string to byte[]-
     *
     * @param hexString input hex string
     * @return byte[] of the given hex string
     */
    byte[] fromHexString(String hexString);

    /**
     * Creates an HMAC
     *
     * @param data raw data
     * @param key secret key
     * @return HMAC according to https://datatracker.ietf.org/doc/html/rfc2104
     */
    String calculateHMAC(String data, String key);
}
