package dev.fitko.fitconnect.api.services.crypto;

import com.nimbusds.jose.JWEObject;
import com.nimbusds.jose.jwk.RSAKey;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.ApiAttachment;
import dev.fitko.fitconnect.api.domain.model.metadata.data.Data;
import dev.fitko.fitconnect.api.domain.model.submission.SubmitSubmission;
import dev.fitko.fitconnect.api.exceptions.internal.DecryptionException;
import dev.fitko.fitconnect.api.exceptions.internal.EncryptionException;
import java.io.InputStream;

/**
 * A service that allows to encrypt and decrypt {@link Data} and {@link ApiAttachment}s of a {@link
 * SubmitSubmission} via JSON-Web-Encryption.
 *
 * @see <a href="https://datatracker.ietf.org/doc/html/rfc7516">JSON-Web-Encryption</a>
 */
public interface CryptoService {

    /**
     * Decrypts a JWE encrypted string with the given private key.
     *
     * @param privateKey RSA private key for decryption of JWE
     * @param encryptedData serialized encrypted JWE Hex string
     * @return decrypted JWE byte array
     * @throws DecryptionException if the payload cannot be decrypted or there was an issue with the
     *     key
     */
    byte[] decryptToBytes(RSAKey privateKey, String encryptedData) throws DecryptionException;

    /**
     * Encrypts an object with the given public key.
     *
     * @param encryptionKey RSA public key for encryption of string payload
     * @param obj object that should be encrypted
     * @param contentType MIME content type of the object to encrypt
     * @return Hex string of the encrypted JWE object
     * @throws EncryptionException if the payload cannot be encrypted or there was an issue with the
     *     key
     */
    String encryptObject(RSAKey encryptionKey, Object obj, String contentType);

    /**
     * Encrypts a byte[] payload with the given public key.
     *
     * @param encryptionKey RSA public key the payload is encrypted with
     * @param bytes byte[] of the data that should be encrypted
     * @param contentType MIME content type of the string to encrypt
     * @return Hex string of the encrypted JWE object
     * @throws EncryptionException if the payload cannot be encrypted or there was an issue with the
     *     key
     */
    String encryptBytes(RSAKey encryptionKey, byte[] bytes, String contentType) throws EncryptionException;

    /**
     * Encrypts an input-stream payload with the given public key.
     *
     * @param encryptionKey RSA public key the payload is encrypted with
     * @param inputStream input-stream of the data that should be encrypted
     * @param contentType MIME content type of the string to encrypt
     * @return Hex string of the encrypted JWE object
     * @throws EncryptionException if the payload cannot be encrypted or there was an issue with the
     *     key
     */
    JWEObject encryptInputStream(RSAKey encryptionKey, InputStream inputStream, String contentType)
            throws EncryptionException;

    /**
     * Creates a message digest hash from the given input byte[]
     *
     * @param data byte[] that should be hashed
     * @return hex encoded string of the hashed data
     */
    String hashBytes(byte[] data);

    /**
     * Creates a message digest hash from the given input-stream
     *
     * @param inputStream that should be hashed
     * @return hex encoded string of the hashed data
     */
    String hashStream(InputStream inputStream);
}
