package dev.fitko.fitconnect.api.domain.validation;

import lombok.AllArgsConstructor;
import lombok.Data;

/** Represents the result of a virus scan operation. */
@Data
@AllArgsConstructor
public class VirusScanResult {

    /** The raw result string from the virus scanner engine. */
    private String result;

    /** The signature (name) of any detected malware or an empty string if no malware is found. */
    private String signature;

    /** Flag indicating whether malware was detected during the scan. */
    private boolean infected;

    /** Flag indicating that was an error during the scan. */
    private boolean scanFailed;

    /**
     * Creates a VirusScanResult instance for a clean scan result.
     *
     * @param result The raw result string from the virus scanner
     * @return A new VirusScanResult instance indicating no malware was detected
     */
    public static VirusScanResult ofClean(String result) {
        return new VirusScanResult(result, "", false, false);
    }

    /**
     * Creates a VirusScanResult instance for a clean scan result.
     *
     * @return A new VirusScanResult instance indicating no malware was detected
     */
    public static VirusScanResult ofClean() {
        return new VirusScanResult("", "", false, false);
    }

    /**
     * Creates a VirusScanResult instance for a failed scan result.
     *
     * @param errorMessage The raw error result string from the virus scanner
     * @return A new VirusScanResult instance indicating no malware was detected
     */
    public static VirusScanResult ofScanFailed(String errorMessage) {
        return new VirusScanResult(errorMessage, "", false, true);
    }

    /**
     * Creates a VirusScanResult instance for an infected scan result.
     *
     * @param result The raw result string from the virus scanner
     * @param signature The signature (name) of the detected malware
     * @return A new VirusScanResult instance indicating malware was detected
     */
    public static VirusScanResult ofInfected(String result, String signature) {
        return new VirusScanResult(result, signature, true, false);
    }
}
