package dev.fitko.fitconnect.api.domain.validation;

import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.exceptions.internal.ValidationException;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import lombok.Getter;

@Getter
public class ValidationContext {
    private final List<ValidationResult> validationResults;
    private final UUID destinationId;
    private final UUID caseId;
    private final AuthenticationTags authenticationTags;
    private final boolean validateAuthTags;

    private ValidationContext(
            final UUID destinationId,
            final UUID caseId,
            final boolean validateAuthTags,
            final AuthenticationTags authenticationTags) {
        this.destinationId = destinationId;
        this.caseId = caseId;
        this.authenticationTags = authenticationTags;
        validationResults = new ArrayList<>();
        // flag cannot be derived from set authTags because they can still be null if validation is
        // wanted
        this.validateAuthTags = validateAuthTags;
    }

    /**
     * Create context with authTags for destination and caseId
     *
     * @param destinationId unique identifier of the destination
     * @param caseId unique identifier of the case
     * @param authenticationTags the authentication tags of the submission
     * @return ValidationContext
     */
    public static ValidationContext withAuthTagValidation(
            final UUID destinationId, final UUID caseId, final AuthenticationTags authenticationTags) {
        return new ValidationContext(destinationId, caseId, true, authenticationTags);
    }

    /**
     * Create context without authTags for destination and caseId
     *
     * @param destinationId unique identifier of the destination
     * @param caseId unique identifier of the case
     * @return ValidationContext
     */
    public static ValidationContext withoutAuthTagValidation(final UUID destinationId, final UUID caseId) {
        return new ValidationContext(destinationId, caseId, false, null);
    }

    /**
     * Create context without authTags for a caseId
     *
     * @param caseId unique identifier of the case
     * @return ValidationContext
     */
    public static ValidationContext withoutAuthTagValidation(final UUID caseId) {
        return new ValidationContext(null, caseId, false, null);
    }

    public void addResult(final ValidationResult validationResult) {
        if (validationResult.hasError()) {
            validationResults.add(validationResult);
        }
    }

    public void addErrorIfTestFailed(final boolean test, final String message) {
        if (!test) {
            addResult(ValidationResult.error(new ValidationException(message)));
        }
    }

    public void addError(final String message) {
        addResult(ValidationResult.error(new ValidationException(message)));
    }

    public void addError(final Exception exception) {
        addResult(ValidationResult.error(new ValidationException(exception.getMessage(), exception)));
    }

    public UUID getDestinationId() {
        return destinationId;
    }

    public UUID getCaseId() {
        return caseId;
    }

    public AuthenticationTags getAuthenticationTags() {
        return authenticationTags;
    }

    public boolean validateAuthTags() {
        return validateAuthTags;
    }
}
