package dev.fitko.fitconnect.api.domain.subscriber;

import com.nimbusds.jose.jwk.JWK;
import dev.fitko.fitconnect.api.config.ApplicationConfig;
import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.metadata.ContentStructure;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.Purpose;
import dev.fitko.fitconnect.api.domain.model.metadata.data.MimeType;
import dev.fitko.fitconnect.api.domain.subscriber.steps.BuildReplyStep;
import dev.fitko.fitconnect.api.domain.subscriber.steps.OptionalReplyPropertiesStep;
import dev.fitko.fitconnect.api.domain.subscriber.steps.ReplyDataStep;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import lombok.Getter;

@Getter
public final class SendableReply {

    private final UUID caseId;
    private final byte[] data;
    private final URI schemaUri;
    private final MimeType dataMimeType;
    private final List<Attachment> attachments;
    private final JWK replyEncryptionKey;

    private SendableReply(final Builder builder) {
        caseId = builder.getCaseId();
        data = builder.getData();
        schemaUri = builder.getSchemaUri();
        dataMimeType = builder.getDataMimeType();
        attachments = Collections.unmodifiableList(builder.getAttachments());
        replyEncryptionKey = builder.getReplyEncryptionKey();
    }

    /**
     * Creates a new SendableReply builder for a specific case. This method allows you to create a
     * reply for a case by providing just the case ID. You will need to manually set the reply
     * encryption key using {@link ReplyDataStep#setReplyEncryptionKey(JWK)}.
     *
     * @param caseId the unique identifier of the case to reply to
     * @return ReplyDataStep builder to configure the reply data and properties
     */
    public static ReplyDataStep forCase(final UUID caseId) {
        return new Builder(caseId);
    }

    @Getter
    private static final class Builder implements ReplyDataStep, OptionalReplyPropertiesStep, BuildReplyStep {

        private final UUID caseId;
        private final List<Attachment> attachments = new ArrayList<>();
        private byte[] data;
        private URI schemaUri;
        private MimeType dataMimeType;
        private JWK replyEncryptionKey;

        private static final byte[] EMPTY_DATA = new byte[0];

        public Builder(UUID caseId) {
            this.caseId = caseId;
        }

        public OptionalReplyPropertiesStep setJsonData(final String replyData, final URI replyDataSchemaUri) {
            setJsonData(replyData.getBytes(StandardCharsets.UTF_8), replyDataSchemaUri);
            return this;
        }

        @Override
        public OptionalReplyPropertiesStep setJsonData(byte[] data, URI schemaUri) {
            this.dataMimeType = MimeType.APPLICATION_JSON;
            this.data = getValidatedData(data);
            this.schemaUri = schemaUri;
            return this;
        }

        public OptionalReplyPropertiesStep setXmlData(final String replyData, final URI replyDataSchemaUri) {
            setXmlData(replyData.getBytes(StandardCharsets.UTF_8), replyDataSchemaUri);
            return this;
        }

        @Override
        public OptionalReplyPropertiesStep setXmlData(byte[] data, URI schemaUri) {
            this.dataMimeType = MimeType.APPLICATION_XML;
            this.data = getValidatedData(data);
            this.schemaUri = schemaUri;
            return this;
        }

        @Override
        public ReplyDataStep setReplyEncryptionKey(JWK replyEncryptionKey) {
            this.replyEncryptionKey = replyEncryptionKey;
            return this;
        }

        @Override
        public OptionalReplyPropertiesStep addAttachments(final List<Attachment> attachments) {
            if (attachments != null && !attachments.isEmpty()) {
                this.attachments.addAll(attachments);
            }
            return this;
        }

        @Override
        public OptionalReplyPropertiesStep addAttachment(final Attachment attachment) {
            this.attachments.add(attachment);
            return this;
        }

        @Override
        public SendableReply build() {
            return new SendableReply(this);
        }

        /**
         * Sets the reply data either as attachment or as byte[], depending on allowed size. <br>
         * In case of an attachment, the data will be transferred with {@link Purpose#DATA} and the data
         * within the {@link ContentStructure} will zero bytes.
         */
        private byte[] getValidatedData(byte[] data) {
            if (data.length > ApplicationConfig.MAX_DATA_SIZE_IN_BYTE) {
                attachments.add(Attachment.fromSubmissionData(data, dataMimeType));
                return EMPTY_DATA;
            }
            return data;
        }
    }
}
