package dev.fitko.fitconnect.api.domain.subscriber;

import static dev.fitko.fitconnect.api.domain.model.event.EventPayload.forRejectEvent;

import dev.fitko.fitconnect.api.FitConnectService;
import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.event.Event;
import dev.fitko.fitconnect.api.domain.model.event.EventPayload;
import dev.fitko.fitconnect.api.domain.model.event.problems.Problem;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.Purpose;
import dev.fitko.fitconnect.api.domain.model.submission.PublicService;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectSubscriberException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.format.DateTimeParseException;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReceivedSubmission implements Comparable<ReceivedSubmission> {

    private static final Logger LOGGER = LoggerFactory.getLogger(ReceivedSubmission.class);

    private final FitConnectService subscriber;
    private final ReceivedSubmissionData receivedData;

    public ReceivedSubmission(final FitConnectService subscriber, final ReceivedSubmissionData receivedData) {
        this.subscriber = subscriber;
        this.receivedData = receivedData;
    }

    /**
     * Send an accept-submission {@link Event} to confirm that the handed in submission was accepted.
     *
     * @param problems optional {@link Problem}s
     * @see <a
     *     href="https://docs.fitko.de/fit-connect/docs/getting-started/event-log/events">FIT-Connect
     *     Events</a>
     */
    public void acceptSubmission(final Problem... problems) {
        try {
            subscriber.acceptSubmission(getEventPayload(problems));
        } catch (final Exception e) {
            throw new FitConnectSubscriberException(
                    "Accepting submission failed", e.getCause() != null ? e.getCause() : e);
        }
    }

    /**
     * Send a reject-submission {@link Event} to confirm that the handed in submission was rejected.
     *
     * @param rejectionProblems list of {@link Problem}s that give more detailed information on why
     *     the submission was rejected
     * @see <a
     *     href="https://docs.fitko.de/fit-connect/docs/getting-started/event-log/events">FIT-Connect
     *     Events</a>
     */
    public void rejectSubmission(final List<Problem> rejectionProblems) {
        try {
            subscriber.rejectSubmission(forRejectEvent(receivedData.getSubmission(), rejectionProblems));
        } catch (final Exception e) {
            throw new FitConnectSubscriberException(
                    "Rejecting submission failed", e.getCause() != null ? e.getCause() : e);
        }
    }

    /**
     * Gets the decrypted data as string. Use {@link #getDataMimeType()} to determine the data format
     * (json or xml).
     *
     * @return data as string
     */
    public String getDataAsString() {
        return new String(receivedData.getData(), StandardCharsets.UTF_8);
    }

    /**
     * Gets the decrypted data as byte[]. Use {@link #getDataMimeType()} to determine the data format
     * (json or xml).
     *
     * @return data as byte[]
     */
    public byte[] getDataAsBytes() {
        return receivedData.getData();
    }

    /**
     * Gets data mime-type.
     *
     * @return mimetype string
     */
    public String getDataMimeType() {
        return getMetadata()
                .getContentStructure()
                .getData()
                .getSubmissionSchema()
                .getMimeType()
                .value();
    }

    /**
     * Gets the data schema reference.
     *
     * @return data schema as URI
     */
    public URI getDataSchemaUri() {
        return getMetadata()
                .getContentStructure()
                .getData()
                .getSubmissionSchema()
                .getSchemaUri();
    }

    /**
     * Gets the submission service type.
     *
     * @return {@link PublicService} object
     */
    public PublicService getServiceType() {
        return receivedData.getSubmission().getPublicService();
    }

    /**
     * Gets the region of the service the submission was handed in to.
     *
     * @return optional string of the region
     */
    public Optional<String> getRegion() {
        return Optional.ofNullable(receivedData.getSubmission().getRegion());
    }

    /**
     * Access the list of decrypted attachments. <br>
     * NOTE: Attachments with {@link Purpose#DATA} are omitted since this attachment type is only used
     * internally to transfer large data sets.
     *
     * @return list of {@link Attachment}
     */
    public List<Attachment> getAttachments() {
        return receivedData.getAttachments();
    }

    /**
     * Gets the submission {@link Metadata}.
     *
     * @return metadata
     */
    public Metadata getMetadata() {
        return receivedData.getMetadata();
    }

    /**
     * Gets the application date when the submission was handed in as date "yyyy-MM-dd" This date
     * doesn't necessarily match with the technical date (iat) the submission was sent via
     * FIT-Connect.
     *
     * @return an optional LocalDate of the application date
     * @see #getSubmittedAt()
     */
    public Optional<LocalDate> getApplicationDate() {

        final Metadata metadata = getMetadata();

        if (metadata == null
                || metadata.getAdditionalReferenceInfo() == null
                || metadata.getAdditionalReferenceInfo().getApplicationDate() == null) {
            return Optional.empty();
        }

        try {
            return Optional.of(
                    LocalDate.parse(metadata.getAdditionalReferenceInfo().getApplicationDate()));
        } catch (final DateTimeParseException e) {
            LOGGER.error("Invalid application date format {}", e.getMessage());
            return Optional.empty();
        }
    }

    /**
     * Gets the submission caseId.
     *
     * @return caseId as UUID
     */
    public UUID getCaseId() {
        return receivedData.getSubmission().getCaseId();
    }

    /**
     * Gets the submission destinationId.
     *
     * @return destinationId as UUID
     */
    public UUID getDestinationId() {
        return receivedData.getSubmission().getDestinationId();
    }

    /**
     * Gets the submission submissionId.
     *
     * @return submissionId as UUID
     */
    public UUID getSubmissionId() {
        return receivedData.getSubmission().getSubmissionId();
    }

    /**
     * Get the date when the submission was submitted. This is the technical issued-at (iat)
     * timestamp.
     *
     * @return timestamp as {@link Date}
     * @see #getApplicationDate()
     */
    public Date getSubmittedAt() {
        return receivedData.getSubmissionStatus().getIssuedAt();
    }

    private EventPayload getEventPayload(final Problem[] problems) {
        if (getAttachments().isEmpty()) {
            return EventPayload.forAcceptEvent(receivedData.getSubmission(), problems);
        }
        return EventPayload.forAcceptEventWithAttachments(
                receivedData.getSubmission(), receivedData.getAttachmentAuthTags(), problems);
    }

    @Override
    public int compareTo(final ReceivedSubmission other) {
        return getSubmittedAt().compareTo(other.getSubmittedAt());
    }
}
