package dev.fitko.fitconnect.api.domain.sender.steps.unencrypted;

import dev.fitko.fitconnect.api.config.defaults.MetadataVersion;
import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.callback.Callback;
import dev.fitko.fitconnect.api.domain.model.metadata.AdditionalReferenceInfo;
import dev.fitko.fitconnect.api.domain.model.metadata.AuthenticationInformation;
import dev.fitko.fitconnect.api.domain.model.metadata.payment.PaymentInformation;
import dev.fitko.fitconnect.api.domain.model.metadata.v2.Author;
import dev.fitko.fitconnect.api.domain.model.metadata.v2.DataSet;
import dev.fitko.fitconnect.api.domain.model.reply.replychannel.ReplyChannel;
import dev.fitko.fitconnect.api.domain.sender.SendableSubmission;
import dev.fitko.fitconnect.api.domain.subscriber.ReceivedSubmission;
import java.net.URI;
import java.time.LocalDate;
import java.util.List;
import java.util.UUID;

public interface OptionalPropertiesStep {

    /**
     * Add a list of {@link Attachment}s to the sendable submission.
     *
     * @param attachments that are added to the existing attachments
     * @return next step to set more optional properties
     */
    OptionalPropertiesStep addAttachments(List<Attachment> attachments);

    /**
     * Add a new {@link Attachment} to the sendable submission.
     *
     * @param attachment that is added to a list of attachments
     * @return next step to set more optional properties
     */
    OptionalPropertiesStep addAttachment(Attachment attachment);

    /**
     * Add a reply channel. The {@link ReplyChannel} object provides static factories for all
     * available options:
     *
     * <ul>
     *   <li>ReplyChannel.ofEmail(...)
     *   <li>ReplyChannel.ofDeMail(...)
     *   <li>ReplyChannel.ofFink(...)
     *   <li>ReplyChannel.ofElster(...)
     *   <li>ReplyChannel.ofFitConnect(...)
     * </ul>
     *
     * @param replyChannel a configured reply channel
     * @return next step to set more optional properties
     * @see <a
     *     href="https://docs.fitko.de/fit-connect/docs/metadata/replyChannel#r%C3%BCckkanal-replychannel">FIT-Connect
     *     documentation on reply channels</a>
     */
    OptionalPropertiesStep setReplyChannel(ReplyChannel replyChannel);

    /**
     * Set {@link AuthenticationInformation} that are necessary for the submission.
     *
     * @param authenticationInformation list of {@link AuthenticationInformation}
     * @return next step to set more optional properties
     * @see <a
     *     href="https://docs.fitko.de/fit-connect/docs/metadata/authenticationInformation">FIT-Connect
     *     documentation on authentication information</a>
     */
    OptionalPropertiesStep setAuthenticationInformation(List<AuthenticationInformation> authenticationInformation);

    /**
     * Set {@link PaymentInformation} that contains information on the payment service. If the
     * optional field is not set the service was not billed yet or is free of charge.
     *
     * @param paymentInformation the payment object containing transaction details
     * @return next step to set more optional properties
     * @see <a href="https://docs.fitko.de/fit-connect/docs/metadata/paymentInformation">FIT-Connect
     *     documentation on payment information</a>
     */
    OptionalPropertiesStep setPaymentInformation(PaymentInformation paymentInformation);

    /**
     * Sets the application date when the submission was handed in. This date can differ from the
     * technical date (issued at) the submission was sent via FIT-Connect.
     *
     * @param applicationDate the date when the submission was handed in
     * @return next step to set more optional properties
     * @see ReceivedSubmission#getApplicationDate()
     * @see ReceivedSubmission#getSubmittedAt()
     */
    OptionalPropertiesStep setApplicationDate(LocalDate applicationDate);

    /**
     * Sets the id.bund.de applicationId.
     *
     * @param idBundDeApplicationId unique identifier of the applicationId
     * @return next step to set more optional properties
     */
    OptionalPropertiesStep setIdBundDeApplicationId(UUID idBundDeApplicationId);

    /**
     * Sets the sender-reference within the {@link AdditionalReferenceInfo}.
     *
     * @param senderReference reference to a process in the sending system
     * @return next step to set more optional properties
     */
    OptionalPropertiesStep setSenderReference(String senderReference);

    /**
     * Set {@link Callback} to get asynchronously notified about submission and reply status updates
     * without polling.
     *
     * @param callbackUri URI that should be called to notify the sender
     * @param callbackSecret secret to verify the callback
     * @return next step to set more optional properties
     * @see <a
     *     href="https://docs.fitko.de/fit-connect/docs/details/callbacks#callback-url">FIT-Connect
     *     documentation on callbacks</a>
     */
    OptionalPropertiesStep setCallback(URI callbackUri, String callbackSecret);

    /**
     * Add the new submission to an existing case.
     *
     * @param caseId unique identifier of the submissions case
     * @return next step to set more optional properties
     */
    OptionalPropertiesStep setCase(UUID caseId);

    /**
     * Set an array that can contains data records, each corresponding to an externalized schema.
     * Each element contains both the actual data and a reference to its associated schema.
     * <p>
     * <b>Note:</b> This method is only available from metadata version 2.x.x.
     *
     * @param dataSets list of data sets for external schemas
     * @return next step to set more optional properties
     */
    OptionalPropertiesStep setDataSets(List<DataSet> dataSets);

    /**
     * Set the author information for the submission.
     * <p>
     * <b>Note:</b> This method is only available from metadata version 2.x.x.
     *
     * @param author the author of the submission
     * @return next step to set more optional properties
     */
    OptionalPropertiesStep setAuthor(Author author);

    /**
     *  Prefer a specific metadata schema major version for the submission (1.x or 2.x).
     *  If the destination supports the preferred major version, the latest available major version is used.
     *  If the destination doesn't support the preferred major version, the latest supported metadata version from the destination will be used.
     *
     * If not specified, the default version from the configuration will be used.
     *
     * @param metadataVersion the version of the metadata schema to use
     * @return next step to set more optional properties
     *
     * @see MetadataVersion#V1
     * @see MetadataVersion#V2
     */
    OptionalPropertiesStep preferMetadataVersion(MetadataVersion metadataVersion);

    /**
     * Constructs a new sendable submission from the data that is set.
     *
     * @return SendableSubmission
     */
    SendableSubmission build();
}
