package dev.fitko.fitconnect.api.domain.sender;

import static dev.fitko.fitconnect.api.config.ApplicationConfig.MAX_DATA_SIZE_IN_BYTE;
import static dev.fitko.fitconnect.api.domain.model.metadata.data.MimeType.APPLICATION_JSON;
import static java.util.Optional.ofNullable;

import dev.fitko.fitconnect.api.config.defaults.MetadataVersion;
import dev.fitko.fitconnect.api.config.defaults.SchemaConfig;
import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.callback.Callback;
import dev.fitko.fitconnect.api.domain.model.metadata.AdditionalReferenceInfo;
import dev.fitko.fitconnect.api.domain.model.metadata.AuthenticationInformation;
import dev.fitko.fitconnect.api.domain.model.metadata.ContentStructure;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.Purpose;
import dev.fitko.fitconnect.api.domain.model.metadata.data.MimeType;
import dev.fitko.fitconnect.api.domain.model.metadata.data.SubmissionSchema;
import dev.fitko.fitconnect.api.domain.model.metadata.payment.PaymentInformation;
import dev.fitko.fitconnect.api.domain.model.metadata.v2.Author;
import dev.fitko.fitconnect.api.domain.model.metadata.v2.DataSet;
import dev.fitko.fitconnect.api.domain.model.reply.replychannel.ReplyChannel;
import dev.fitko.fitconnect.api.domain.model.submission.PublicService;
import dev.fitko.fitconnect.api.domain.sender.steps.unencrypted.*;
import dev.fitko.fitconnect.api.domain.zbp.AuthorKeyPair;
import dev.fitko.fitconnect.api.domain.zbp.message.CreateMessage;
import dev.fitko.fitconnect.api.domain.zbp.state.CreateState;
import dev.fitko.fitconnect.client.zbp.ZBPEnvelopeBuilder;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import lombok.Getter;

@Getter
public final class SendableSubmission {

    private final UUID destinationId;
    private final UUID caseId;
    private final byte[] data;
    private final SubmissionSchema submissionSchema;
    private final List<Attachment> attachments;
    private final PublicService serviceType;
    private final String serviceRegion;
    private final List<AuthenticationInformation> authenticationInformation;
    private final AdditionalReferenceInfo additionalReferenceInfo;
    private final PaymentInformation paymentInformation;
    private final ReplyChannel replyChannel;
    private final Callback callback;
    private final MetadataVersion metadataVersion;
    private final List<DataSet> dataSets;
    private final Author author;

    private SendableSubmission(final Builder builder) {
        destinationId = builder.getDestinationId();
        caseId = builder.getCaseId();
        data = builder.getData();
        submissionSchema = new SubmissionSchema(builder.getSchemaUri(), builder.getDataMimeType());
        attachments = Collections.unmodifiableList(builder.getAttachments());
        serviceType = builder.getServiceType();
        serviceRegion = builder.getServiceRegion();
        authenticationInformation = builder.getAuthenticationInformation();
        replyChannel = builder.getReplyChannel();
        paymentInformation = builder.getPaymentInformation();
        callback = builder.getCallback();
        additionalReferenceInfo = builder.getAdditionalReferenceInfo();
        dataSets = builder.getDataSets();
        author = builder.getAuthor();
        metadataVersion = ofNullable(builder.getMetadataVersion()).orElse(MetadataVersion.V2);
    }

    public static DestinationStep Builder() {
        return new Builder();
    }

    @Getter
    private static final class Builder
            implements DestinationStep, ServiceTypeStep, DataStep, OptionalPropertiesStep, BuildStep {

        private UUID destinationId;
        private UUID caseId;
        private final List<Attachment> attachments = new ArrayList<>();
        private byte[] data;
        private URI schemaUri;
        private MimeType dataMimeType;
        private PublicService serviceType;
        private String serviceRegion;
        private List<AuthenticationInformation> authenticationInformation;
        private PaymentInformation paymentInformation;
        private ReplyChannel replyChannel;
        private AdditionalReferenceInfo additionalReferenceInfo = new AdditionalReferenceInfo();
        private Callback callback;
        private List<DataSet> dataSets;
        private Author author;
        private MetadataVersion metadataVersion;

        private static final byte[] EMPTY_DATA = new byte[0];

        private Builder() {}

        @Override
        public ServiceTypeStep setDestination(final UUID destinationId) {
            this.destinationId = destinationId;
            return this;
        }

        @Override
        public OptionalPropertiesStep addAttachments(final List<Attachment> attachments) {
            if (attachments != null && !attachments.isEmpty()) {
                this.attachments.addAll(attachments);
            }
            return this;
        }

        @Override
        public OptionalPropertiesStep addAttachment(final Attachment attachment) {
            if (attachment != null) {
                this.attachments.add(attachment);
            }
            return this;
        }

        @Override
        public OptionalPropertiesStep setReplyChannel(final ReplyChannel replyChannel) {
            this.replyChannel = replyChannel;
            return this;
        }

        @Override
        public OptionalPropertiesStep setJsonData(final String jsonData, final URI schemaUri) {
            setJsonData(jsonData.getBytes(StandardCharsets.UTF_8), schemaUri);
            return this;
        }

        @Override
        public OptionalPropertiesStep setJsonData(byte[] data, URI schemaUri) {
            this.dataMimeType = APPLICATION_JSON;
            this.schemaUri = schemaUri;
            this.data = getValidatedData(data);
            return this;
        }

        @Override
        public OptionalPropertiesStep setXmlData(final String xmlData, final URI schemaUri) {
            setXmlData(xmlData.getBytes(StandardCharsets.UTF_8), schemaUri);
            return this;
        }

        @Override
        public OptionalPropertiesStep setXmlData(byte[] data, URI schemaUri) {
            this.dataMimeType = MimeType.APPLICATION_XML;
            this.schemaUri = schemaUri;
            this.data = getValidatedData(data);
            return this;
        }

        @Override
        public OptionalPropertiesStep setZBPMessage(CreateMessage message, AuthorKeyPair authorKeyPair) {
            data = ZBPEnvelopeBuilder.fromAuthorPayload(message, authorKeyPair);
            dataMimeType = APPLICATION_JSON;
            this.schemaUri = SchemaConfig.ZBP_ADAPTER_SCHEMA.getSchemaUri();
            return this;
        }

        @Override
        public OptionalPropertiesStep setZBPState(CreateState state, AuthorKeyPair authorKeyPair) {
            data = ZBPEnvelopeBuilder.fromAuthorPayload(state, authorKeyPair);
            dataMimeType = APPLICATION_JSON;
            this.schemaUri = SchemaConfig.ZBP_ADAPTER_SCHEMA.getSchemaUri();
            return this;
        }

        @Override
        public DataStep setServiceType(final String serviceIdentifier, final String serviceName) {
            this.serviceType = new PublicService(serviceName, serviceIdentifier);
            return this;
        }

        @Override
        public DataStep setServiceTypeWithRegion(String serviceIdentifier, String serviceName, String serviceRegion) {
            this.serviceType = new PublicService(serviceName, serviceIdentifier);
            this.serviceRegion = serviceRegion;
            return this;
        }

        @Override
        public OptionalPropertiesStep setAuthenticationInformation(
                final List<AuthenticationInformation> authenticationInformation) {
            this.authenticationInformation = authenticationInformation;
            return this;
        }

        @Override
        public OptionalPropertiesStep setPaymentInformation(final PaymentInformation paymentInformation) {
            this.paymentInformation = paymentInformation;
            return this;
        }

        @Override
        public OptionalPropertiesStep setApplicationDate(final LocalDate applicationDate) {
            additionalReferenceInfo = additionalReferenceInfo.withApplicationDate(applicationDate.toString());
            return this;
        }

        @Override
        public OptionalPropertiesStep setIdBundDeApplicationId(UUID idBundDeApplicationId) {
            additionalReferenceInfo = additionalReferenceInfo.withIdBundDeApplicationId(idBundDeApplicationId);
            return this;
        }

        @Override
        public OptionalPropertiesStep setSenderReference(String senderReference) {
            additionalReferenceInfo = additionalReferenceInfo.withSenderReference(senderReference);
            return this;
        }

        @Override
        public OptionalPropertiesStep setCallback(final URI callbackUri, final String callbackSecret) {
            callback = new Callback(callbackUri, callbackSecret);
            return this;
        }

        @Override
        public OptionalPropertiesStep setCase(final UUID caseId) {
            this.caseId = caseId;
            return this;
        }

        @Override
        public OptionalPropertiesStep setDataSets(List<DataSet> dataSets) {
            this.dataSets = dataSets;
            return this;
        }

        @Override
        public OptionalPropertiesStep setAuthor(Author author) {
            this.author = author;
            return this;
        }

        @Override
        public OptionalPropertiesStep preferMetadataVersion(MetadataVersion metadataVersion) {
            this.metadataVersion = metadataVersion;
            return this;
        }

        @Override
        public SendableSubmission build() {
            return new SendableSubmission(this);
        }

        /**
         * Sets the submission data either as attachment or as byte[], depending on allowed size. <br>
         * In case of an attachment, the data will be transferred with {@link Purpose#DATA} and the data
         * within the {@link ContentStructure} will zero bytes.
         */
        private byte[] getValidatedData(byte[] data) {
            if (data.length > MAX_DATA_SIZE_IN_BYTE) {
                attachments.add(Attachment.fromSubmissionData(data, dataMimeType));
                return EMPTY_DATA;
            }
            return data;
        }
    }
}
