package dev.fitko.fitconnect.api.domain.sender;

import dev.fitko.fitconnect.api.FitConnectService;
import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.event.problems.Problem;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.Purpose;
import dev.fitko.fitconnect.api.domain.model.reply.AcceptReply;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;

public class ReceivedReply {

    private final FitConnectService fitConnectService;
    private final ReceivedReplyData receivedReplyData;

    public ReceivedReply(final FitConnectService fitConnectService, ReceivedReplyData receivedReplyData) {
        this.fitConnectService = fitConnectService;
        this.receivedReplyData = receivedReplyData;
    }

    public void acceptReply(final Problem... problems) {
        fitConnectService.acceptReply(
                receivedReplyData.getReply().getReplyId(),
                new AcceptReply(Arrays.asList(problems), receivedReplyData.getAuthenticationTags()));
    }

    public void rejectReply(final List<Problem> problems) {
        fitConnectService.rejectReply(receivedReplyData.getReply().getReplyId(), problems);
    }

    /**
     * Get decrypted reply data as string.
     *
     * @return data as string
     */
    public String getDataAsString() {
        return new String(receivedReplyData.getData(), StandardCharsets.UTF_8);
    }

    /**
     * Get decrypted reply data as byte[]
     *
     * @return data as bytes
     */
    public byte[] getDataAsByteArray() {
        return receivedReplyData.getData();
    }

    /**
     * Gets the reply data mime-type.
     *
     * @return mimetype string
     */
    public String getDataMimeType() {
        return receivedReplyData
                .getMetadata()
                .getContentStructure()
                .getData()
                .getSubmissionSchema()
                .getMimeType()
                .value();
    }

    /**
     * Gets the reply data schema reference.
     *
     * @return data schema as URI
     */
    public URI getDataSchemaUri() {
        return receivedReplyData
                .getMetadata()
                .getContentStructure()
                .getData()
                .getSubmissionSchema()
                .getSchemaUri();
    }

    /**
     * Access the list of decrypted attachments. <br>
     * NOTE: Attachments with {@link Purpose#DATA} are omitted since this attachment type is only used
     * internally to transfer large data sets.
     *
     * @return list of {@link Attachment}
     */
    public List<Attachment> getAttachments() {
        return receivedReplyData.getAttachments();
    }

    /**
     * Gets the reply {@link Metadata}.
     *
     * @return metadata
     */
    public Metadata getMetadata() {
        return receivedReplyData.getMetadata();
    }

    /**
     * Gets the reply caseId.
     *
     * @return caseId as UUID
     */
    public UUID getCaseId() {
        return receivedReplyData.getReply().getCaseId();
    }

    /**
     * Gets the reply id.
     *
     * @return replyId as UUID
     */
    public UUID getReplyId() {
        return receivedReplyData.getReply().getReplyId();
    }
}
