package dev.fitko.fitconnect.api.domain.routing;

import java.util.Objects;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import lombok.AllArgsConstructor;
import lombok.Value;

/**
 * Builds a new search request for a service identifier and AT MAX. ONE OTHER search criterion (ars
 * | ags | areaId).
 */
@Value
@AllArgsConstructor
public class DestinationSearch {

    String leikaKey;
    String ars;
    String ags;
    String areaId;

    int offset;
    int limit;

    public static MandatoryProperties Builder() {
        return new Builder();
    }

    public interface MandatoryProperties {

        /**
         * Leika Key of the requested service.
         *
         * @param leikaKey service identifier
         * @return Builder
         * @throws IllegalArgumentException if the leika key pattern is not matching
         */
        OptionalProperties withLeikaKey(final String leikaKey) throws IllegalArgumentException;
    }

    public interface OptionalProperties {

        /**
         * Official municipal code of the place.
         *
         * @param ars amtlicher regionalschlüssel
         * @return Builder
         * @throws IllegalArgumentException if the ars key pattern is not matching
         */
        Pagination withArs(final String ars) throws IllegalArgumentException;

        /**
         * Official regional key of the area.
         *
         * @param ags amtlicher gemeindeschlüssel
         * @return Builder
         * @throws IllegalArgumentException if the ags key pattern is not matching
         */
        Pagination withAgs(final String ags) throws IllegalArgumentException;

        /**
         * ID of the area. This ID can be determined via the routing clients <code>findAreas</code>
         * search.
         *
         * @param areaId id of the area
         * @return Builder
         * @throws IllegalArgumentException if the area id pattern is not matching
         */
        Pagination withAreaId(final String areaId) throws IllegalArgumentException;

        /**
         * Start position of the subset of the result set. Default is 0.
         *
         * @param offset start of the subset
         * @return Builder
         * @throws IllegalArgumentException if the offset is a negative number
         */
        Builder withOffset(final int offset) throws IllegalArgumentException;

        /**
         * Max. size of the subset of the result set. Maximum is 500. Default is 100.
         *
         * @param limit max. entries in the subset
         * @return Builder
         * @throws IllegalArgumentException if the limit is > 500
         */
        Builder withLimit(final int limit) throws IllegalArgumentException;

        /**
         * Construct the search request.
         *
         * @return DestinationSearch
         */
        DestinationSearch build() throws IllegalArgumentException;
    }

    public interface Pagination {

        /**
         * Start position of the subset of the result set. Default is 0.
         *
         * @param offset start of the subset
         * @return Builder
         * @throws IllegalArgumentException if the offset is a negative number
         */
        Pagination withOffset(final int offset) throws IllegalArgumentException;

        /**
         * Max. size of the subset of the result set. Maximum is 500. Default is 100.
         *
         * @param limit max. entries in the subset
         * @return Builder
         * @throws IllegalArgumentException if the limit is > 500
         */
        Pagination withLimit(final int limit) throws IllegalArgumentException;

        /**
         * Construct the search request.
         *
         * @return DestinationSearch
         */
        DestinationSearch build() throws IllegalArgumentException;
    }

    public interface BuildSearch {
        /**
         * Construct the search request.
         *
         * @return DestinationSearch
         */
        DestinationSearch build() throws IllegalArgumentException;
    }

    private static class Builder implements MandatoryProperties, OptionalProperties, Pagination, BuildSearch {

        private static final Pattern AGS_PATTERN = Pattern.compile("^(\\d{2}|\\d{3}|\\d{5}|\\d{8})$");
        private static final Pattern ARS_PATTERN = Pattern.compile("^(\\d{2}|\\d{3}|\\d{5}|\\d{9}|\\d{12})$");
        private static final Pattern AREA_ID_PATTERN = Pattern.compile("^\\d{1,}");
        private static final Pattern LEIKA_KEY_PATTERN = Pattern.compile("^99\\d{12}$");

        private String leikaKey;
        private String ars;
        private String ags;
        private String areaId;
        private int offset = 0;
        private int limit = 100;

        @Override
        public OptionalProperties withLeikaKey(final String leikaKey) throws IllegalArgumentException {
            if (!LEIKA_KEY_PATTERN.matcher(leikaKey).matches()) {
                throw new IllegalArgumentException("Leika key does not match allowed pattern " + LEIKA_KEY_PATTERN);
            }
            this.leikaKey = leikaKey;
            return this;
        }

        @Override
        public Pagination withArs(final String ars) throws IllegalArgumentException {
            if (!ARS_PATTERN.matcher(ars).matches()) {
                throw new IllegalArgumentException("ARS key does not match allowed pattern " + ARS_PATTERN);
            }
            this.ars = ars;
            return this;
        }

        @Override
        public Pagination withAgs(final String ags) throws IllegalArgumentException {
            if (!AGS_PATTERN.matcher(ags).matches()) {
                throw new IllegalArgumentException("AGS key does not match allowed pattern " + AGS_PATTERN);
            }
            this.ags = ags;
            return this;
        }

        @Override
        public Pagination withAreaId(final String areaId) throws IllegalArgumentException {
            if (!AREA_ID_PATTERN.matcher(areaId).matches()) {
                throw new IllegalArgumentException("AreaId key does not match allowed pattern " + AREA_ID_PATTERN);
            }
            this.areaId = areaId;
            return this;
        }

        @Override
        public Builder withOffset(final int offset) throws IllegalArgumentException {
            if (limit < 0) {
                throw new IllegalArgumentException("offset must be positive");
            }
            this.offset = offset;
            return this;
        }

        @Override
        public Builder withLimit(final int limit) throws IllegalArgumentException {
            if (limit > 500) {
                throw new IllegalArgumentException("limit must not be > 500");
            }
            this.limit = limit;
            return this;
        }

        @Override
        public DestinationSearch build() throws IllegalArgumentException {
            if (leikaKey == null) {
                throw new IllegalArgumentException("leikaKey must not be null");
            }
            if (Stream.of(areaId, ags, ars).allMatch(Objects::isNull)) {
                throw new IllegalArgumentException(
                        "at least one regional search criterion (areaId, ars or ags) is mandatory");
            }
            return new DestinationSearch(leikaKey, ars, ags, areaId, offset, limit);
        }
    }
}
