package dev.fitko.fitconnect.api.domain.model.reply.replychannel;

import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.nimbusds.jose.jwk.JWK;
import java.util.List;
import java.util.UUID;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;

@Data
@NoArgsConstructor
@AllArgsConstructor
@JsonInclude(JsonInclude.Include.NON_NULL)
public class ReplyChannel {

    private DeMail deMail;

    private Elster elster;

    private Email email;

    private Fink fink;

    private FitConnect fitConnect;

    private IdBundDeMailbox idBundDeMailbox;

    private ReplyChannel(final Fink fink) {
        this.fink = fink;
    }

    private ReplyChannel(final Email email) {
        this.email = email;
    }

    private ReplyChannel(final DeMail deMail) {
        this.deMail = deMail;
    }

    private ReplyChannel(final Elster elster) {
        this.elster = elster;
    }

    private ReplyChannel(final FitConnect fitConnect) {
        this.fitConnect = fitConnect;
    }

    private ReplyChannel(final IdBundDeMailbox idBundDeMailbox) {
        this.idBundDeMailbox = idBundDeMailbox;
    }

    /**
     * Set a postbox address of an interoperable service account (FINK) as reply channel.
     *
     * @param finkPostboxRef unique identifier of the fink postbox
     * @param host url of the service account
     * @return the {@link ReplyChannel}
     */
    public static ReplyChannel ofFink(final String finkPostboxRef, final String host) {
        return new ReplyChannel(new Fink(finkPostboxRef, host));
    }

    /**
     * Set email with PGP-key as reply channel.
     *
     * @param address email address
     * @param pgpPublicKey public pgp key
     * @return the {@link ReplyChannel}
     * @see <a href="https://docs.fitko.de/fit-connect/docs/metadata/replyChannel/">Reply Channel
     *     Creation</a>
     */
    public static ReplyChannel ofEmailWithPgp(final String address, final String pgpPublicKey) {
        return new ReplyChannel(new Email(address, true, pgpPublicKey));
    }

    /**
     * Set email as reply channel.
     *
     * @param address email address
     * @return the {@link ReplyChannel}
     */
    public static ReplyChannel ofEmail(final String address) {
        return new ReplyChannel(new Email(address, false, null));
    }

    /**
     * Set deMail as reply channel.
     *
     * @param address email address
     * @return the {@link ReplyChannel}
     */
    public static ReplyChannel ofDeMail(final String address) {
        return new ReplyChannel(new DeMail(address));
    }

    /**
     * Set an elster account as reply channel.
     *
     * @param accountId elster account id
     * @param deliveryTicket elster delivery ticket
     * @param reference elster reference
     * @return the {@link ReplyChannel}
     */
    public static ReplyChannel ofElster(final String accountId, final String deliveryTicket, final String reference) {
        return new ReplyChannel(new Elster(accountId, deliveryTicket, reference));
    }

    /**
     * Set FIT-Connect as a reply channel.
     *
     * @param encryptionKey JWK of the public encryption key the reply sender can encrypt the reply
     *     data with
     * @param processStandards list of URIs that reference a data processing standard
     * @return the {@link ReplyChannel}
     */
    public static ReplyChannel ofFitConnect(final JWK encryptionKey, final List<String> processStandards) {
        return new ReplyChannel(new FitConnect(EncryptionPublicKey.fromJwk(encryptionKey), processStandards));
    }

    /**
     * Set a mailboxId for id.bund.de as a reply channel.
     *
     * @param mailboxId postkorbHandle of the bund id mailbox
     * @return the {@link ReplyChannel}
     */
    public static ReplyChannel ofIdBundDeMailbox(final UUID mailboxId) {
        return new ReplyChannel(new IdBundDeMailbox(mailboxId));
    }

    /**
     * Checks if reply channel type is DeMail.
     *
     * @return true or false
     */
    public boolean isDeMail() {
        return this.deMail != null;
    }

    /**
     * Checks if reply channel type is Elster.
     *
     * @return true or false
     */
    public boolean isElster() {
        return this.elster != null;
    }

    /**
     * Checks if reply channel type is eMail.
     *
     * @return true or false
     */
    public boolean isEMail() {
        return this.email != null;
    }

    /**
     * Checks if reply channel type is Fink.
     *
     * @return true or false
     */
    public boolean isFink() {
        return this.fink != null;
    }

    /**
     * Checks if reply channel type is FIT-Connect.
     *
     * @return true or false
     */
    public boolean isFitConnect() {
        return this.fitConnect != null;
    }

    /**
     * Checks if reply channel type is IdBundDeId.
     *
     * @return true or false
     */
    public boolean isIdBundDeMailbox() {
        return this.idBundDeMailbox != null;
    }

    // Explicit getters with @JsonGetter to prevent Jackson from creating duplicate fields
    @JsonGetter("deMail")
    public DeMail getDeMail() {
        return deMail;
    }

    @JsonGetter("elster")
    public Elster getElster() {
        return elster;
    }

    @JsonGetter("eMail")
    public Email getEmail() {
        return email;
    }

    @JsonGetter("fink")
    public Fink getFink() {
        return fink;
    }

    @JsonGetter("fitConnect")
    public FitConnect getFitConnect() {
        return fitConnect;
    }

    @JsonGetter("idBundDeMailbox")
    public IdBundDeMailbox getIdBundDeMailbox() {
        return idBundDeMailbox;
    }
}
