package dev.fitko.fitconnect.api.domain.model.event;

import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.problems.Problem;
import dev.fitko.fitconnect.api.domain.model.submission.Submission;
import dev.fitko.fitconnect.api.domain.model.submission.SubmissionForPickup;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;

@Data
@Builder
@AllArgsConstructor
public class EventPayload {

    private UUID submissionId;
    private UUID destinationId;
    private UUID caseId;

    // auth-tags for validation
    private String metadataAuthTag;
    private String dataAuthTag;
    private Map<UUID, String> attachmentAuthTags;

    private List<Problem> problems;

    public static EventPayload forRejectEvent(final SubmissionForPickup submission, final List<Problem> problems) {
        return EventPayload.builder()
                .submissionId(submission.getSubmissionId())
                .destinationId(submission.getDestinationId())
                .caseId(submission.getCaseId())
                .problems(problems)
                .build();
    }

    public static EventPayload forRejectEvent(final Submission submission, final List<Problem> problems) {
        return EventPayload.builder()
                .submissionId(submission.getSubmissionId())
                .destinationId(submission.getDestinationId())
                .caseId(submission.getCaseId())
                .problems(problems)
                .build();
    }

    public static EventPayload forAcceptEvent(final Submission submission, final Problem... problems) {
        return EventPayload.builder()
                .submissionId(submission.getSubmissionId())
                .destinationId(submission.getDestinationId())
                .caseId(submission.getCaseId())
                .dataAuthTag(AuthenticationTags.getAuthTagFromJWT(submission.getEncryptedData()))
                .metadataAuthTag(AuthenticationTags.getAuthTagFromJWT(submission.getEncryptedMetadata()))
                .attachmentAuthTags(Collections.emptyMap())
                .problems(Arrays.asList(problems))
                .build();
    }

    public static EventPayload forAcceptEventWithAttachments(
            final Submission submission, final Map<UUID, String> attachmentAuthTags, final Problem... problems) {
        return EventPayload.builder()
                .submissionId(submission.getSubmissionId())
                .destinationId(submission.getDestinationId())
                .caseId(submission.getCaseId())
                .dataAuthTag(AuthenticationTags.getAuthTagFromJWT(submission.getEncryptedData()))
                .metadataAuthTag(AuthenticationTags.getAuthTagFromJWT(submission.getEncryptedMetadata()))
                .attachmentAuthTags(attachmentAuthTags)
                .problems(Arrays.asList(problems))
                .build();
    }
}
