package dev.fitko.fitconnect.api.domain.limits.destination;

import static dev.fitko.fitconnect.core.utils.Preconditions.checkNotNull;

import com.fasterxml.jackson.annotation.JsonInclude;
import dev.fitko.fitconnect.api.domain.limits.Limit;
import java.time.Instant;
import java.util.UUID;
import lombok.Data;
import lombok.NoArgsConstructor;

@Data
@NoArgsConstructor
@JsonInclude(JsonInclude.Include.NON_NULL)
public class LimitChangeRequest {

    private UUID id;
    private Limit values;
    private String contactEmail;
    private String requestReason;
    private String decisionComment;
    private ChangeRequestStatus status;
    private Instant receivedAt;
    private Instant decidedAt;

    public LimitChangeRequest(Limit values, String requestReason, ChangeRequestStatus status, String contactEmail) {
        checkNotNull("status", status);
        this.values = values;
        this.requestReason = requestReason;
        this.status = status;
        this.contactEmail = contactEmail;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private Limit values;
        private String contactEmail;
        private String requestReason;
        private ChangeRequestStatus status;

        Builder() {}

        /**
         * Limits that apply when attachments are sent to this destination.
         *
         * @param values the {@link Limit}s to be updated
         * @return the {@link Builder}
         */
        public Builder values(Limit values) {
            this.values = values;
            return this;
        }

        /**
         * Email address of a contact person who would like to be informed about the result of the
         * request.
         *
         * @param contactEmail email address of the contact
         * @return the {@link Builder}
         */
        public Builder contactEmail(String contactEmail) {
            this.contactEmail = contactEmail;
            return this;
        }

        /**
         * Brief explanation as to why an increase in the limitation is desired.
         *
         * @param requestReason the reason as string
         * @return the {@link Builder}
         */
        public Builder requestReason(String requestReason) {
            this.requestReason = requestReason;
            return this;
        }

        /**
         * Sets the status for a request. PENDING for new requests or to edit an existing request.
         * WITHDRAWN to take back a request.
         *
         * @param status the requested status
         * @return the {@link Builder}
         */
        public Builder status(ChangeRequestStatus status) {
            this.status = status;
            return this;
        }

        public LimitChangeRequest build() {
            return new LimitChangeRequest(this.values, this.requestReason, this.status, this.contactEmail);
        }
    }
}
