package dev.fitko.fitconnect.api.domain.crypto;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.nimbusds.jose.jwk.JWK;
import dev.fitko.fitconnect.api.domain.model.jwk.ApiJwk;
import lombok.Value;

/**
 * Represents a pair of JSON Web Keys (JWK) in the FIT-Connect system. This class holds both the
 * public and private keys used for:
 *
 * <ul>
 *   <li>Encryption/decryption of data
 *   <li>Digital signatures
 *   <li>Key exchange
 * </ul>
 *
 * The keys are stored in JWK format according to RFC 7517.
 *
 * @see <a href="https://docs.fitko.de/fit-connect/docs/getting-started/get-started">FIT-Connect
 *     Getting Started</a>
 * @see <a href="https://docs.fitko.de/fit-connect/docs/getting-started/encryption">Encryption
 *     Documentation</a>
 * @see <a href="https://tools.ietf.org/html/rfc7517">RFC 7517 - JSON Web Key (JWK)</a>
 */
@Value
public class JWKPair {

    JWK publicKey;
    JWK privateKey;

    private static final ObjectMapper MAPPER = new ObjectMapper();

    /**
     * Gets the pair's public key as {@link ApiJwk}.
     *
     * @return {@link ApiJwk} representation of the public key
     */
    public ApiJwk getPublicApiJwk() {
        try {
            return MAPPER.readValue(publicKey.toJSONString(), ApiJwk.class);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }
}
