package dev.fitko.fitconnect.api.config.virusscan;

import dev.fitko.fitconnect.core.validation.virusscan.deamon.ClamAVDeamonConfig;
import dev.fitko.fitconnect.core.validation.virusscan.process.ClamAVProcessConfig;
import lombok.Builder;
import lombok.Data;

/**
 * Configuration for virus scanner settings.
 *
 * <p>It allows configuring both ClamAV daemon and process-based scanning implementations:
 *
 * <p>
 *
 * <ul>
 *   <li><strong>ClamAV Daemon Scanner:</strong> Connects to a running ClamAV daemon service via TCP
 *       socket
 *   <li><strong>ClamAV Process Scanner:</strong> Executes ClamAV as a separate process for each
 *       scan
 * </ul>
 *
 * <p>
 *
 * @see dev.fitko.fitconnect.core.validation.virusscan.VirusScannerFactory
 * @see dev.fitko.fitconnect.core.validation.virusscan.ClamAVDeamonScanner
 * @see dev.fitko.fitconnect.core.validation.virusscan.ClamAVProcessScanner
 * @see dev.fitko.fitconnect.api.services.validation.VirusScanService
 */
@Data
@Builder
public class VirusScannerConfig {

    /**
     * Configuration for ClamAV daemon scanner.
     *
     * <p>This configuration controls the behavior of the ClamAV daemon scanner, which connects to a
     * running ClamAV daemon service via TCP socket for virus scanning operations.
     *
     * <p>
     *
     * @see dev.fitko.fitconnect.core.validation.virusscan.deamon.ClamAVDeamonConfig
     * @see dev.fitko.fitconnect.core.validation.virusscan.ClamAVDeamonScanner
     */
    @Builder.Default
    private ClamAVDeamonConfig clamAVDeamonConfig = ClamAVDeamonConfig.defaultConfig();

    /**
     * Configuration for ClamAV process scanner.
     *
     * <p>This configuration controls the behavior of the ClamAV process scanner, which executes
     * ClamAV as a separate process for each virus scanning operation.
     *
     * <p>
     *
     * @see dev.fitko.fitconnect.core.validation.virusscan.process.ClamAVProcessConfig
     * @see dev.fitko.fitconnect.core.validation.virusscan.ClamAVProcessScanner
     */
    @Builder.Default
    private ClamAVProcessConfig clamAVProcessConfig = ClamAVProcessConfig.defaultConfig();

    /**
     * Creates a default virus scanner configuration with default settings for both ClamAV daemon and
     * process scanners.
     *
     * @return a default virus scanner configuration
     * @see ClamAVDeamonConfig#defaultConfig()
     * @see ClamAVProcessConfig#defaultConfig()
     */
    public static VirusScannerConfig defaultConfig() {
        return VirusScannerConfig.builder().build();
    }

    /**
     * Creates a daemon mode configuration.
     *
     * @param daemonHost the hostname or IP address of the ClamAV daemon
     * @param daemonPort the port number of the ClamAV daemon
     * @return a daemon-only configuration
     */
    public static VirusScannerConfig daemonOnly(String daemonHost, int daemonPort) {
        return VirusScannerConfig.builder()
                .clamAVDeamonConfig(ClamAVDeamonConfig.builder()
                        .host(daemonHost)
                        .port(daemonPort)
                        .build())
                .clamAVProcessConfig(null)
                .build();
    }

    /**
     * Creates a process mode configuration.
     *
     * @param executablePath the path to the ClamAV executable
     * @return a process-only configuration
     */
    public static VirusScannerConfig processOnly(String executablePath) {
        return VirusScannerConfig.builder()
                .clamAVDeamonConfig(null)
                .clamAVProcessConfig(ClamAVProcessConfig.of(executablePath))
                .build();
    }
}
