package dev.fitko.fitconnect.api.config.chunking;

import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import java.nio.file.Path;

public class AttachmentChunkingConfig {

    public static final String DEFAULT_ATTACHMENT_FOLDER_NAME = "fit-connect-attachments";
    public static final String TEMP_BUFFERED_FILE_PREFIX = "temp_fit_connect_attachment_";

    private final boolean chunkAllAttachments;
    private final ChunkSize chunkSizeInMB;
    private final String attachmentStoragePath;

    public AttachmentChunkingConfig(
            final boolean chunkAllAttachments, final ChunkSize chunkSizeInMB, final String attachmentStoragePath) {
        this.chunkAllAttachments = chunkAllAttachments;
        this.chunkSizeInMB = chunkSizeInMB;
        this.attachmentStoragePath = attachmentStoragePath;
    }

    private AttachmentChunkingConfig(
            final boolean chunkAllAttachments, final ChunkSize chunkSizeInMB, final Path attachmentStoragePath) {
        this(
                chunkAllAttachments,
                chunkSizeInMB,
                attachmentStoragePath == null ? null : attachmentStoragePath.toString());
    }

    public AttachmentChunkingConfig() {
        this(false, null, "");
    }

    public static AttachmentConfigBuilder builder() {
        return new AttachmentConfigBuilder();
    }

    /**
     * Checks if attachment chunking for all attachments, including in-memory attachments, is active.
     * Per default large attachments will be chunked automatically.
     *
     * <ul>
     *   <li>If true all attachments will be chunked
     *   <li>If false only large attachments attachments will be chunked
     * </ul>
     *
     * @return true or false
     */
    public boolean isChunkAllAttachments() {
        return chunkAllAttachments;
    }

    /**
     * Size of the chunks the attachment will be split into.
     *
     * @return chunk size in bytes
     */
    public int getChunkSizeInBytes() {
        return chunkSizeInMB.getSizeInBytes();
    }

    /**
     * Size of the chunks the attachment will be split into.
     *
     * @return chunk size in MB (^10)
     */
    public int getChunkSizeInMB() {
        return chunkSizeInMB.getSizeInMB();
    }

    /**
     * Gets the chunk-size object.
     *
     * @return {@link ChunkSize}
     */
    public ChunkSize getChunkSizeObject() {
        return chunkSizeInMB;
    }

    /**
     * Path where the sdk stores attachment chunks for up- and download.
     *
     * @return storage path, null if not set
     */
    public Path getAttachmentStoragePath() {
        if (attachmentStoragePath == null || attachmentStoragePath.isEmpty()) {
            return null;
        }
        return Path.of(attachmentStoragePath);
    }

    public static class AttachmentConfigBuilder {
        private boolean chunkAllAttachments;
        private ChunkSize chunkSize;
        private Path attachmentStoragePath;

        AttachmentConfigBuilder() {}

        /**
         * Activate attachment chunking for all attachments, including in-memory attachments. Per
         * default, chunking is only active for large attachments that don't fit into memory.
         *
         * @param chunkAllAttachments chunking for all attachment types on == true or off == false
         * @return the builder
         * @see Attachment#fromLargeAttachment
         */
        public AttachmentConfigBuilder chunkAllAttachments(final boolean chunkAllAttachments) {
            this.chunkAllAttachments = chunkAllAttachments;
            return this;
        }

        /**
         * Set the chunk size in MB.
         *
         * <p>If no value is set, the allowed limit will be used (see attachment limits).
         *
         * <p><i>NOTE: be aware that the actual size of a chunk on upload to FIT-Connect will be approx.
         * 33% larger due to the encryption overhead. When choosing a chunk-size, consider an
         * appropriate value. </i>
         *
         * @param sizeInMB size of a file chunk in MegaByte (^10)
         * @return the builder
         * @see <a
         *     href="https://docs.fitko.de/fit-connect/docs/apis/submission-api#get-/v2/destinations/-destinationId-/limits">Attachment-Limits
         *     SubmissionAPI</a>
         */
        public AttachmentConfigBuilder chunkSizeInMB(final int sizeInMB) {
            this.chunkSize = ChunkSize.ofMB(sizeInMB);
            return this;
        }

        /**
         * Path where the sdk stores attachment chunks for up- and download. If no path is set a folder
         * in "java.io.tmpdir" is used.
         *
         * @param attachmentStoragePath path where attachment chunks and files are stored
         * @return the builder
         * @see AttachmentChunkingConfig#DEFAULT_ATTACHMENT_FOLDER_NAME
         */
        public AttachmentConfigBuilder attachmentStoragePath(final Path attachmentStoragePath) {
            this.attachmentStoragePath = attachmentStoragePath;
            return this;
        }

        /**
         * Build a new config object.
         *
         * @return AttachmentConfig
         */
        public AttachmentChunkingConfig build() {
            return new AttachmentChunkingConfig(chunkAllAttachments, chunkSize, attachmentStoragePath);
        }
    }
}
