package dev.fitko.fitconnect.api.config;

import static dev.fitko.fitconnect.core.utils.Strings.isNotNullOrEmpty;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.jwk.RSAKey;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Optional;
import lombok.Getter;

@Getter
public class ZBPCertConfig {

    private final RSAKey clientPrivateKey;
    private final String clientCertificate;
    private final Optional<String> zbpServerCertificate;

    /**
     * Constructor to set ZBP client certificate and private key. This uses the default system trust
     * store to verify the identity of the ZBP-Server (this usually is what you want).
     *
     * @param privateKeyPath path to sender private key pem
     * @param clientCertPath path to sender public certificate pem
     */
    public ZBPCertConfig(Path privateKeyPath, Path clientCertPath) {
        this.clientPrivateKey = parseKeyFromPath(privateKeyPath);
        this.clientCertificate = readPemStringFromPath(clientCertPath);
        this.zbpServerCertificate = Optional.empty();
    }

    /**
     * Constructor to set ZBP client certificate and private. This uses the default system trust store
     * to verify the identity of the ZBP-Server (this usually is what you want).
     *
     * @param privateKeyAsPem sender private key pem
     * @param clientCertAsPem sender public certificate pem
     */
    public ZBPCertConfig(String privateKeyAsPem, String clientCertAsPem) {
        this.clientPrivateKey = getRsaKeyFromPem(privateKeyAsPem);
        this.clientCertificate = clientCertAsPem;
        this.zbpServerCertificate = Optional.empty();
    }

    /**
     * Constructor to set ZBP client certificate, private key, and server certificate to trust. This
     * is for advanced use cases. The default system trust store is ignored and only the specified
     * server certificate is trusted.
     *
     * @param clientPrivateKey sender private key as {@link RSAKey}
     * @param clientCertificateAsPem sender public certificate pem
     * @param zbpServerCertificateAsPem the id.bund.de server ssl certificate as pem
     */
    public ZBPCertConfig(RSAKey clientPrivateKey, String clientCertificateAsPem, String zbpServerCertificateAsPem) {
        this.clientPrivateKey = clientPrivateKey;
        this.clientCertificate = clientCertificateAsPem;
        this.zbpServerCertificate = Optional.ofNullable(zbpServerCertificateAsPem);
    }

    public boolean hasValidConfiguration() {
        return clientPrivateKey != null && isNotNullOrEmpty(clientCertificate);
    }

    private RSAKey parseKeyFromPath(Path privateKeyPath) {
        return getRsaKeyFromPem(readPemStringFromPath(privateKeyPath));
    }

    private RSAKey getRsaKeyFromPem(String privateKeyPem) {
        try {
            return RSAKey.parseFromPEMEncodedObjects(privateKeyPem).toRSAKey();
        } catch (JOSEException e) {
            throw new RuntimeException("Could not parse key", e);
        }
    }

    private String readPemStringFromPath(Path pemPath) {
        try {
            return Files.readString(pemPath);
        } catch (IOException e) {
            throw new RuntimeException("Could not read pem from path " + pemPath, e);
        }
    }
}
