package dev.fitko.fitconnect.api.config;

import static java.lang.String.valueOf;

import java.util.Arrays;
import java.util.Objects;
import lombok.Getter;
import lombok.Value;

/**
 * Version class to compare SemVer versions.
 * Supports versions like "1.0.0" and pre-releases like "2.0.0-rc2".
 */
@Getter
public final class Version implements Comparable<Version> {

    public static final char VERSION_SEPARATOR_CHAR = '.';
    public static final String VERSION_SEPARATOR_STRING = "\\.";
    public static final char PRE_RELEASE_SEPARATOR_CHAR = '-';

    private final int major;
    private final int minor;
    private final int patch;
    private final String preRelease;

    public Version(String version) {
        final VersionComponents components = parseVersion(version);
        this.major = components.major;
        this.minor = components.minor;
        this.patch = components.patch;
        this.preRelease = components.preRelease;
    }

    public Version(int major, int minor, int patch) {
        this.major = major;
        this.minor = minor;
        this.patch = patch;
        this.preRelease = null;
    }

    public Version(int major, int minor, int patch, String preRelease) {
        this.major = major;
        this.minor = minor;
        this.patch = patch;
        this.preRelease = preRelease;
    }

    /**
     * Gets the original version as string
     *
     * @return version with major, minor, patch, and optional pre-release
     */
    public String getVersionAsString() {
        String baseVersion = major + valueOf(VERSION_SEPARATOR_CHAR) + minor + VERSION_SEPARATOR_CHAR + patch;
        if (preRelease != null) {
            return baseVersion + PRE_RELEASE_SEPARATOR_CHAR + preRelease;
        }
        return baseVersion;
    }

    /**
     * Test if the version is greater than a given other version.
     * Pe-release versions have lower precedence than release versions.
     *
     * @param otherVersion the other version to compare with
     * @return true if version > otherVersion
     */
    public boolean isGreaterThan(Version otherVersion) {
        return this.compareTo(otherVersion) == 1;
    }

    /**
     * Test if the version is greater or equal than a given other version.
     *
     * @param otherVersion the other version to compare with
     * @return true if version >= otherVersion
     */
    public boolean isGreaterOrEqualThan(Version otherVersion) {
        return isGreaterThan(otherVersion) || equals(otherVersion);
    }

    private VersionComponents parseVersion(String version) {
        if (version == null || version.trim().isEmpty()) {
            throw new IllegalArgumentException("Version must not be null or empty");
        }

        String trimmedVersion = version.trim();
        String[] parts = trimmedVersion.split(String.valueOf(PRE_RELEASE_SEPARATOR_CHAR));
        String baseVersion = parts[0];
        String preReleasePart = parts.length > 1 ? parts[1] : null;

        long dotCount =
                baseVersion.chars().filter(c -> c == VERSION_SEPARATOR_CHAR).count();
        if (dotCount != 2) {
            throw new IllegalArgumentException("Version " + version + " is not in major.minor.patch format");
        }

        int[] components = Arrays.stream(baseVersion.split(VERSION_SEPARATOR_STRING))
                .mapToInt(Integer::parseInt)
                .toArray();

        return new VersionComponents(components[0], components[1], components[2], preReleasePart);
    }

    @Override
    public String toString() {
        return getVersionAsString();
    }

    @Value
    @Getter
    private static class VersionComponents {
        int major;
        int minor;
        int patch;
        String preRelease;
    }

    @Override
    public int compareTo(Version other) {
        if (other == null) {
            throw new NullPointerException("Cannot compare with null version");
        }

        int majorComparison = Integer.compare(this.major, other.major);
        if (majorComparison != 0) {
            return majorComparison;
        }

        int minorComparison = Integer.compare(this.minor, other.minor);
        if (minorComparison != 0) {
            return minorComparison;
        }

        int patchComparison = Integer.compare(this.patch, other.patch);
        if (patchComparison != 0) {
            return patchComparison;
        }

        return comparePreRelease(this.preRelease, other.preRelease);
    }

    private int comparePreRelease(String thisPreRelease, String otherPreRelease) {
        if (thisPreRelease == null && otherPreRelease == null) {
            return 0;
        }

        if (thisPreRelease == null && otherPreRelease != null) {
            return 1;
        }

        if (thisPreRelease != null && otherPreRelease == null) {
            return -1;
        }

        return thisPreRelease.compareTo(otherPreRelease);
    }

    @Override
    public boolean equals(Object other) {
        Version otherVersion = (Version) other;
        return this.major == otherVersion.major
                && this.minor == otherVersion.minor
                && this.patch == otherVersion.patch
                && Objects.equals(this.preRelease, otherVersion.preRelease);
    }
}
