package dev.fitko.fitconnect.api.config;

import static java.util.Objects.requireNonNull;

import com.nimbusds.jose.jwk.JWK;
import java.util.List;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;

@Getter
@NoArgsConstructor
public class SubscriberConfig {
    private String clientId;
    private String clientSecret;
    private String privateSigningKeyPath;
    private List<String> privateDecryptionKeyPaths;
    private SubscriberKeys subscriberKeys;

    @Getter
    @AllArgsConstructor
    public static class SubscriberKeys {
        private JWK privateSigningKey;
        private List<JWK> privateDecryptionKeys;
    }

    public SubscriberConfig(
            String clientId,
            String clientSecret,
            String privateSigningKeyPath,
            List<String> privateDecryptionKeyPaths) {
        this.clientId = clientId;
        this.clientSecret = clientSecret;
        this.privateSigningKeyPath = privateSigningKeyPath;
        this.privateDecryptionKeyPaths = privateDecryptionKeyPaths;
    }

    private SubscriberConfig(String clientId, String clientSecret, SubscriberKeys subscriberKeys) {
        requireNonNull(clientId, "clientId must not be null");
        requireNonNull(clientSecret, "client secret must not be null");
        requireNonNull(subscriberKeys, "subscriber keys must not be null");

        this.clientId = clientId;
        this.clientSecret = clientSecret;
        this.subscriberKeys = subscriberKeys;
    }

    public static SubscriberConfigBuilder builder() {
        return new SubscriberConfigBuilder();
    }

    public static class SubscriberConfigBuilder {
        private String clientId;
        private String clientSecret;
        private JWK privateSigningKey;
        private List<JWK> privateDecryptionKeys;
        private String privateSigningKeyPath;
        private List<String> privateDecryptionKeyPaths;

        SubscriberConfigBuilder() {}

        /**
         * The subscribers clientID.
         *
         * @param clientId clientId as string
         * @return SubscriberConfigBuilder
         */
        public SubscriberConfigBuilder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        /**
         * The subscribers client secret.
         *
         * @param clientSecret secret as string
         * @return SubscriberConfigBuilder
         */
        public SubscriberConfigBuilder clientSecret(String clientSecret) {
            this.clientSecret = clientSecret;
            return this;
        }

        /**
         * JWK of the private signature key.
         *
         * @param privateSigningKey the private signature key as {@link JWK}
         * @return SubscriberConfigBuilder
         * @see JWK
         */
        public SubscriberConfigBuilder privateSigningKey(JWK privateSigningKey) {
            this.privateSigningKey = privateSigningKey;
            return this;
        }

        /**
         * Path to the private signature key.
         *
         * @param privateSigningKeyPath signature key path
         * @return SubscriberConfigBuilder
         */
        public SubscriberConfigBuilder privateSigningKeyPath(String privateSigningKeyPath) {
            this.privateSigningKeyPath = privateSigningKeyPath;
            return this;
        }

        /**
         * List of JWKs of private decryption keys.
         *
         * @param privateDecryptionKeys list of private decryption keys as {@link JWK}
         * @return SubscriberConfigBuilder
         * @see JWK
         */
        public SubscriberConfigBuilder privateDecryptionKeys(List<JWK> privateDecryptionKeys) {
            this.privateDecryptionKeys = privateDecryptionKeys;
            return this;
        }

        /**
         * List of paths to the private decryption keys.
         *
         * @param privateDecryptionKeyPaths list of strings with paths to the private decryption keys
         * @return SubscriberConfigBuilder
         */
        public SubscriberConfigBuilder privateDecryptionKeyPaths(List<String> privateDecryptionKeyPaths) {
            this.privateDecryptionKeyPaths = privateDecryptionKeyPaths;
            return this;
        }

        public SubscriberConfig build() {
            if (privateSigningKey != null && privateDecryptionKeys != null) {
                return new SubscriberConfig(
                        clientId, clientSecret, new SubscriberKeys(privateSigningKey, privateDecryptionKeys));
            } else {
                return new SubscriberConfig(clientId, clientSecret, privateSigningKeyPath, privateDecryptionKeyPaths);
            }
        }
    }
}
