package dev.fitko.fitconnect.api.config;

import static dev.fitko.fitconnect.api.config.defaults.Environments.getAvailableEnvironmentNames;
import static dev.fitko.fitconnect.api.config.defaults.SchemaConfig.DESTINATION_SCHEMA_VERSION;
import static dev.fitko.fitconnect.api.config.defaults.SchemaConfig.METADATA_SCHEMA_PATH;
import static dev.fitko.fitconnect.api.config.defaults.SchemaConfig.SET_SCHEMA_PATH;

import dev.fitko.fitconnect.api.config.chunking.AttachmentChunkingConfig;
import dev.fitko.fitconnect.api.config.defaults.Environments;
import dev.fitko.fitconnect.api.config.http.HttpConfig;
import dev.fitko.fitconnect.api.config.virusscan.VirusScannerConfig;
import dev.fitko.fitconnect.api.config.virusscan.VirusScannerMode;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectInitialisationException;
import java.net.URI;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.ToString;
import lombok.With;
import lombok.experimental.Accessors;

@Getter
@Builder
@Accessors(chain = true)
@ToString
@NoArgsConstructor
@AllArgsConstructor
public class ApplicationConfig {

    public static final Version SET_SCHEMA_WRITE_VERSION = new Version("1.2.2");

    public static final Version METADATA_SCHEMA_V1_WRITE_VERSION = new Version("1.5.0");

    public static final Version METADATA_SCHEMA_V2_WRITE_VERSION = new Version("2.0.0");

    public static final Version MIN_METADATA_VERSION_WITH_CHUNKING_SUPPORT = new Version("1.3.0");

    public static final int MAX_DATA_SIZE_IN_BYTE = 13 << 10 << 10; // 13 MB

    public static final Pattern LEIKA_KEY_PATTERN =
            Pattern.compile("^urn:[a-z0-9][a-z0-9-]{0,31}:[a-z0-9()+,.:=@;$_!*'%/?#-]+$");

    /*************** Section is customizable via YAML **************/
    private SenderConfig senderConfig;

    private SubscriberConfig subscriberConfig;

    private EnvironmentName activeEnvironment;

    @Builder.Default
    private HttpConfig httpConfig = new HttpConfig();

    @Builder.Default
    private AttachmentChunkingConfig attachmentChunkingConfig = new AttachmentChunkingConfig();

    @Builder.Default
    private Map<String, String> submissionDataSchemas = new HashMap<>();

    @Builder.Default
    private int concurrentAttachmentStreams = 3;

    @Builder.Default
    private VirusScannerConfig virusScannerConfig = VirusScannerConfig.defaultConfig();

    @Builder.Default
    private VirusScannerMode virusScannerMode = VirusScannerMode.NO_OP;

    @With
    @Builder.Default
    private Map<EnvironmentName, Environment> environments = Environments.getEnvironmentsAsMap();

    /****************************************************************/
    public boolean isAutoRejectEnabled() {
        return getCurrentEnvironment().getEnableAutoReject();
    }

    public boolean isSkipSubmissionDataValidation() {
        return getCurrentEnvironment().getSkipSubmissionDataValidation();
    }

    public boolean isAllowInsecurePublicKey() {
        return getCurrentEnvironment().getAllowInsecurePublicKey();
    }

    public String getSubmissionBaseUrl() {
        return getCurrentEnvironment().getSubmissionBaseUrls().stream()
                .findFirst()
                .orElseThrow(() ->
                        new FitConnectInitialisationException("No submission base url found, expected at least one"));
    }

    public String getDestinationBaseUrl() {
        return getCurrentEnvironment().getDestinationBaseUrl();
    }

    public String getSelfServicePortalBaseUrl() {
        return getCurrentEnvironment().getSelfServicePortalBaseUrl();
    }

    public String getRoutingBaseUrl() {
        return getCurrentEnvironment().getRoutingBaseUrl();
    }

    public String getAuthBaseUrl() {
        return getCurrentEnvironment().getAuthBaseUrl();
    }

    public URI getSetSchemaWriteVersion() {
        return SET_SCHEMA_PATH.getSchemaUriForVersion(SET_SCHEMA_WRITE_VERSION);
    }

    public URI getMetadataSchemaWriteVersion() {
        return METADATA_SCHEMA_PATH.getSchemaUriForVersion(METADATA_SCHEMA_V1_WRITE_VERSION);
    }

    public URI getDestinationSchemaUri() {
        return DESTINATION_SCHEMA_VERSION.getSchemaUri();
    }

    public Environment getCurrentEnvironment() {
        if (activeEnvironment == null) {
            throw new FitConnectInitialisationException(
                    "Environment must not be null. Please set active environment. Available environments are: "
                            + getAvailableEnvironmentNames());
        }
        if (!environments.containsKey(activeEnvironment)) {
            throw new FitConnectInitialisationException("Environment '"
                    + activeEnvironment.getName()
                    + "' not found. Available environments are: "
                    + getAvailableEnvironmentNames());
        }
        return environments.get(activeEnvironment);
    }
}
