package dev.fitko.fitconnect.api;

import com.nimbusds.jose.JWEObject;
import com.nimbusds.jose.jwk.RSAKey;
import dev.fitko.fitconnect.api.domain.limits.submission.SubmissionLimits;
import dev.fitko.fitconnect.api.domain.model.cases.Case;
import dev.fitko.fitconnect.api.domain.model.cases.Cases;
import dev.fitko.fitconnect.api.domain.model.destination.Destination;
import dev.fitko.fitconnect.api.domain.model.destination.PublicDestination;
import dev.fitko.fitconnect.api.domain.model.event.EventLogEntry;
import dev.fitko.fitconnect.api.domain.model.event.EventPayload;
import dev.fitko.fitconnect.api.domain.model.event.Status;
import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.authtags.ValidatedAuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.problems.Problem;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.ApiAttachment;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.AttachmentForValidation;
import dev.fitko.fitconnect.api.domain.model.metadata.data.Data;
import dev.fitko.fitconnect.api.domain.model.reply.AcceptReply;
import dev.fitko.fitconnect.api.domain.model.reply.AnnounceReply;
import dev.fitko.fitconnect.api.domain.model.reply.CreatedReply;
import dev.fitko.fitconnect.api.domain.model.reply.RepliesForPickup;
import dev.fitko.fitconnect.api.domain.model.reply.Reply;
import dev.fitko.fitconnect.api.domain.model.reply.SentReply;
import dev.fitko.fitconnect.api.domain.model.reply.SubmitReply;
import dev.fitko.fitconnect.api.domain.model.submission.AnnounceSubmission;
import dev.fitko.fitconnect.api.domain.model.submission.CreatedSubmission;
import dev.fitko.fitconnect.api.domain.model.submission.SentSubmission;
import dev.fitko.fitconnect.api.domain.model.submission.Submission;
import dev.fitko.fitconnect.api.domain.model.submission.SubmissionForPickup;
import dev.fitko.fitconnect.api.domain.model.submission.SubmissionsForPickup;
import dev.fitko.fitconnect.api.domain.model.submission.SubmitSubmission;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import dev.fitko.fitconnect.api.domain.validation.VirusScanResult;
import dev.fitko.fitconnect.api.exceptions.internal.DecryptionException;
import dev.fitko.fitconnect.api.exceptions.internal.EncryptionException;
import dev.fitko.fitconnect.api.exceptions.internal.EventLogException;
import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import dev.fitko.fitconnect.api.exceptions.internal.VirusScanException;
import java.io.InputStream;
import java.net.URI;
import java.nio.file.Path;
import java.util.List;
import java.util.UUID;

/**
 * Service facade to access all basic SDK services used by sender and subscriber. This is the main
 * entry point for interacting with the FIT-Connect platform.
 *
 * <p>The service provides functionality for: - Sending and receiving submissions - Managing replies
 * - Handling cases and event logs - Cryptographic operations - Validation of data and metadata
 *
 * @see <a href="https://docs.fitko.de/fit-connect/docs/getting-started/get-started">FIT-Connect
 *     Getting Started</a>
 * @see <a href="https://docs.fitko.de/fit-connect/docs/sending/start-submission">Sending
 *     Submissions</a>
 * @see <a
 *     href="https://docs.fitko.de/fit-connect/docs/receiving/process-and-acknowledge">Processing
 *     Submissions</a>
 * @see <a href="https://docs.fitko.de/fit-connect/docs/getting-started/event-log/events">Event Log
 *     Documentation</a>
 */
public interface FitConnectService {

    /**
     * Validates the {@link Metadata} structure against a given JSON schema to ensure its correctness.
     *
     * @param metadata the {@link Metadata} object that is validated
     * @return a {@link ValidationResult}, contains an error if the {@link Metadata} is invalid or
     *     doesn't match the schema
     */
    ValidationResult validateMetadataSchema(Metadata metadata);

    /**
     * Validates the {@link Metadata} structure and contents to ensure its correctness.
     *
     * @param metadata the {@link Metadata} object that is validated
     * @param submission the {@link Submission} of the validated metadata
     * @param authenticationTags the {@link AuthenticationTags} of the validated metadata
     * @return a {@link ValidationResult}, contains an error if the {@link Metadata} is invalid, for
     *     example, doesn't match the schema
     */
    ValidationResult validateSubmissionMetadata(
            Metadata metadata, Submission submission, AuthenticationTags authenticationTags);

    /**
     * Validate json data against a given schema.
     *
     * @param json byte[] of the JSON data to be validated
     * @param schemaUri the URI of the JSON schema
     * @return a {@link ValidationResult}
     */
    ValidationResult validateJsonSchema(byte[] json, URI schemaUri);

    /**
     * Validate XML data against a given schema.
     *
     * @param xml byte[] of the XML data to be validated
     * @param schemaUri the URI of the XML schema
     * @return a {@link ValidationResult}
     */
    ValidationResult validateXmlSchema(byte[] xml, URI schemaUri);

    /**
     * Tests if a given string is well-formed JSON syntax.
     *
     * @param jsonData json byte[] that is tested
     * @return a {@link ValidationResult} with an optional error
     */
    ValidationResult validateJsonFormat(byte[] jsonData);

    /**
     * Tests if a given byte[] is well-formed XML syntax.
     *
     * @param xmlData XML byte[] that is tested
     * @return a {@link ValidationResult} with an optional error
     */
    ValidationResult validateXmlFormat(byte[] xmlData);

    /**
     * Checks if a received callback can be trusted by validating the provided request data
     *
     * @param hmac authentication code provided by the callback
     * @param timestampInSeconds timestamp in seconds provided by the callback
     * @param httpBody HTTP body provided by the callback
     * @param callbackSecret secret owned by the client, which is used to calculate the hmac
     * @return {@code true} if hmac and timestamp provided by the callback meet the required
     *     conditions
     */
    ValidationResult validateCallback(String hmac, Long timestampInSeconds, String httpBody, String callbackSecret);

    /**
     * Validates the attachment structure and contents to ensure its correctness.
     *
     * @param attachmentsForValidation list of attachments containing the hash, decrypted and
     *     encrypted data needed for validation
     * @param authenticationTags the {@link AuthenticationTags} of the validated attachments
     * @return a {@link ValidationResult}, contains an error if the attachment is invalid
     */
    ValidationResult validateAttachments(
            List<AttachmentForValidation> attachmentsForValidation, AuthenticationTags authenticationTags);

    ValidationResult validateReplyMetadata(Metadata metadata, Reply reply, AuthenticationTags authenticationTags);

    /**
     * Validates the {@link Data} structure and contents to ensure its correctness.
     *
     * @param data the unencrypted data as byte[]
     * @param encryptedData the encrypted data of the {@link Submission}
     * @param metadata the {@link Metadata} of the validated data
     * @param authenticationTags the authenticationTags of the validated data
     * @return a {@link ValidationResult}, contains an error if the {@link Metadata} is invalid, e.g.
     *     doesn't match the schema
     */
    ValidationResult validateData(byte[] data, String encryptedData, Metadata metadata, String authenticationTags);

    /**
     * Encrypts the submission data payload (JSON or XML) with JWE (JSON-Web-Encryption).
     *
     * @param publicKey the public encryption key for encryption of {@link Data}
     * @param data the {@link Data} payload that should be encrypted
     * @param contentType MIME content type of the object to encrypt
     * @return the JWE-encrypted {@link Data}
     * @throws EncryptionException if the encryption fails due to invalid key or corrupt data
     */
    String encryptBytes(RSAKey publicKey, byte[] data, String contentType) throws EncryptionException;

    JWEObject encryptInputStream(RSAKey publicKey, InputStream inputStream, String contentType)
            throws EncryptionException;

    /**
     * Encrypts an object with JWE (JSON-Web-Encryption).
     *
     * @param publicKey the public encryption key
     * @param object the data payload that should be encrypted
     * @param contentType MIME content type of the object to encrypt
     * @return the JWE-encrypted {@link Data}
     * @throws EncryptionException if the encryption fails due to invalid key or corrupt data
     */
    String encryptObject(RSAKey publicKey, Object object, String contentType) throws EncryptionException;

    /**
     * Decrypts JWE-encrypted string data.
     *
     * @param privateKey the private key to decrypt the JWE-encrypted string
     * @param encryptedContent JWE encrypted content that should be decrypted
     * @return the decrypted content as byte[]
     */
    byte[] decryptString(RSAKey privateKey, String encryptedContent) throws DecryptionException;

    /**
     * Create message digest (SHA-512) of a given byte[]
     *
     * @param data input bytes that should be hashed
     * @return hex-string representation of hashed byte[]
     */
    String createHash(byte[] data);

    String createHash(InputStream inputStream);

    /**
     * Polls available {@link SubmissionForPickup}s for a given destinationId.
     *
     * @param destinationId restricts the query to a specific destination
     * @param limit number of submissions in result (max. is 500)
     * @param offset position in the dataset
     * @return list of found {@link SubmissionForPickup}s
     */
    SubmissionsForPickup getAvailableSubmissions(UUID destinationId, int offset, int limit) throws RestApiException;

    /**
     * Creates and announces a new {@link AnnounceSubmission}.
     *
     * @param submission with a destinationId, a list of attachmentIds and a serviceType
     * @return the announced submission
     */
    CreatedSubmission announceSubmission(AnnounceSubmission submission);

    /**
     * Send a {@link SubmitSubmission} that includes encrypted {@link Metadata} and {@link Data}
     *
     * @param submission submission to submit
     * @return sent {@link Submission}
     */
    Submission sendSubmission(SubmitSubmission submission);

    /**
     * Uploads encrypted {@link ApiAttachment}s to for given submission id.
     *
     * @param submissionId unique identifier of the submission
     * @param attachmentId unique identifier of the attachment
     * @param encryptedAttachment string of JWE encrypted attachment
     */
    void uploadSubmissionAttachment(UUID submissionId, UUID attachmentId, String encryptedAttachment);

    /**
     * Gets a {@link Submission}.
     *
     * @param submissionId the unique identifier of a {@link Submission}
     * @return the requested {@link Submission}
     */
    Submission getSubmission(UUID submissionId) throws RestApiException;

    /**
     * Uploads an encrypted attachment stream for a submission.
     *
     * @param submissionId unique identifier of the submission
     * @param attachmentId unique identifier of the attachment
     * @param encryptedAttachment input stream containing the JWE encrypted attachment
     * @throws RestApiException if a technical error occurred during the upload
     */
    void uploadSubmissionAttachmentStream(UUID submissionId, UUID attachmentId, InputStream encryptedAttachment)
            throws RestApiException;

    /**
     * Loads encrypted {@link ApiAttachment} for a {@link Submission}.
     *
     * @param submissionId the unique identifier of a {@link Submission}
     * @param attachmentId unique identifier of the attachments
     * @return encrypted JWE string of attachment
     */
    String getSubmissionAttachment(UUID submissionId, UUID attachmentId) throws RestApiException;

    /**
     * Get attachment limits that apply to sending submissions and replies to this destination.
     *
     * @param destinationId unique identifier of the destination
     * @return the {@link SubmissionLimits} for the requested destination
     * @throws RestApiException if the limits aren't available or a technical error occurred
     */
    SubmissionLimits getDestinationAttachmentLimits(UUID destinationId) throws RestApiException;

    /**
     * Get attachment limits that apply to sending submissions and replies related to a case.
     *
     * @param caseId unique identifier of the case
     * @return the {@link SubmissionLimits} for the requested case
     * @throws RestApiException if the limits aren't available or a technical error occurred
     */
    SubmissionLimits getCaseAttachmentLimits(UUID caseId) throws RestApiException;

    /**
     * Sends a confirmation event if the received submission matches all validations.
     *
     * @param eventPayload contains submissionId, caseId, destination and a list of problems
     * @see <a href="https://docs.fitko.de/fit-connect/docs/receiving/process-and-acknowledge">Process
     *     And Acknowledge</a>
     */
    void acceptSubmission(EventPayload eventPayload) throws RestApiException;

    /**
     * Sends a rejection event if the received submission violates any validation rule.
     *
     * @param eventPayload contains submissionId, caseId, destination and a list of problems
     * @see <a href="https://docs.fitko.de/fit-connect/docs/receiving/process-and-acknowledge">Process
     *     And Acknowledge</a>
     */
    void rejectSubmission(EventPayload eventPayload) throws RestApiException;

    /**
     * Gets {@link PublicDestination} by id.
     * This destination is fetched from the SubmissionAPI.
     *
     * @param destinationId unique identifier of a {@link PublicDestination}
     * @return the destination
     */
    PublicDestination getPublicDestination(UUID destinationId);

    /**
     * Gets {@link Destination} by id.
     * This destination is fetched from the DestinationAPI.
     *
     * @param destinationId unique identifier of a {@link Destination}
     * @return the destination
     */
    Destination getPrivateDestination(UUID destinationId);

    /**
     * Retrieves the public encryption key for a destination.
     *
     * @param destinationId unique identifier of the destination
     * @return the public key for encrypting {@link Metadata}, {@link Data} and {@link ApiAttachment}s
     */
    RSAKey getEncryptionKeyForDestination(UUID destinationId);

    /**
     * Retrieves the public encryption key for a destination.
     *
     * @param destination the destination the public key is retrieved for
     * @return the public key for encrypting {@link Metadata}, {@link Data} and {@link ApiAttachment}s
     */
    RSAKey getEncryptionKeyForDestination(PublicDestination destination);

    /**
     * Retrieve the current status of a {@link Submission}.
     *
     * @param sentSubmission sent submission including authentication tags
     * @return {@link Status} the current status
     */
    Status getStatus(SentSubmission sentSubmission) throws RestApiException;

    /**
     * Retrieve the current status of a {@link Reply}.
     *
     * @param reply sent reply including authentication tags
     * @return {@link Status} the current reply status
     */
    Status getStatus(SentReply reply) throws RestApiException;

    /**
     * Retrieve the event log for a given case and destination.
     *
     * @param destinationId unique identifier of the {@link Destination} the log should be retrieved
     *     for
     * @param caseId unique identifier of the case the log should be retrieved for
     * @param authenticationTags
     * @return List of {@link EventLogEntry}s for the given case
     */
    List<EventLogEntry> getEventLogForCase(UUID destinationId, UUID caseId, AuthenticationTags authenticationTags);

    /**
     * Retrieve the event log for a given submission, case and destination.
     *
     * @param destinationId unique identifier of the {@link Destination} the log should be retrieved
     *     for
     * @param caseId unique identifier of the case the log should be retrieved for
     * @param submissionId unique identifier of the submission the log should be retrieved for
     * @param authenticationTags
     * @return List of {@link EventLogEntry}s for the given case
     */
    List<EventLogEntry> getEventLogForSubmission(
            UUID destinationId, UUID caseId, UUID submissionId, AuthenticationTags authenticationTags);

    /**
     * Get event from event-log filtered by a given event and submission.
     *
     * @param submission submission to filter the event-log by
     * @return {@link ValidatedAuthenticationTags} of metadata, data and attachments including a
     *     validation result
     */
    ValidatedAuthenticationTags getSubmissionAuthenticationTags(Submission submission) throws RestApiException;

    /**
     * Get event from event-log filtered by a given event and reply.
     *
     * @param reply reply to filter the event-log by
     * @return {@link ValidatedAuthenticationTags} of metadata, data and attachments including a
     *     validation result
     */
    ValidatedAuthenticationTags getReplyAuthenticationTags(Reply reply) throws RestApiException, EventLogException;

    /**
     * Lists all cases visible to the client.
     *
     * @param limit maximum number of cases to return
     * @param offset number of cases to skip
     * @return list of cases
     * @throws RestApiException if a technical error occurred
     * @throws EventLogException if event log validation failed
     */
    Cases listCases(int limit, int offset) throws RestApiException, EventLogException;

    /**
     * Gets a specific case by its ID.
     *
     * @param caseId unique identifier of the case
     * @return the requested case
     * @throws RestApiException if a technical error occurred
     * @throws EventLogException if event log validation failed
     */
    Case getCase(UUID caseId) throws RestApiException, EventLogException;

    /**
     * Gets a reply by its ID.
     *
     * @param replyId unique identifier of the reply
     * @return the requested reply
     * @throws RestApiException if a technical error occurred
     */
    Reply getReply(UUID replyId) throws RestApiException;

    /**
     * Gets all available replies ready for pickup.
     *
     * @param limit maximum number of replies to return
     * @param offset number of replies to skip
     * @return list of replies available for pickup
     * @throws RestApiException if a technical error occurred
     */
    RepliesForPickup getAvailableReplies(int limit, int offset) throws RestApiException;

    /**
     * Uploads an encrypted attachment for a reply.
     *
     * @param replyId unique identifier of the reply
     * @param attachmentId unique identifier of the attachment
     * @param encryptedAttachment JWE encrypted attachment data
     * @throws RestApiException if a technical error occurred during the upload
     */
    void uploadReplyAttachment(UUID replyId, UUID attachmentId, String encryptedAttachment) throws RestApiException;

    /**
     * Gets an attachment for a reply.
     *
     * @param replyId unique identifier of the reply
     * @param attachmentId unique identifier of the attachment
     * @return the encrypted attachment data
     * @throws RestApiException if a technical error occurred
     */
    String getReplyAttachment(UUID replyId, UUID attachmentId) throws RestApiException;

    /**
     * Submits a reply with encrypted data and metadata.
     *
     * @param replyId unique identifier of the reply
     * @param submitReply reply data to submit
     * @return the sent reply
     * @throws RestApiException if a technical error occurred
     */
    SentReply submitReply(UUID replyId, SubmitReply submitReply) throws RestApiException;

    /**
     * Announces a new reply.
     *
     * @param announceReply reply announcement data
     * @return the created reply
     * @throws RestApiException if a technical error occurred
     */
    CreatedReply announceReply(AnnounceReply announceReply) throws RestApiException;

    /**
     * Accepts a reply with optional problems and authentication tags.
     *
     * @param replyId unique identifier of the reply
     * @param acceptReply acceptance data including optional problems and authentication tags
     * @return confirmation of the acceptance
     * @throws RestApiException if a technical error occurred
     */
    String acceptReply(UUID replyId, AcceptReply acceptReply) throws RestApiException;

    /**
     * Rejects a reply with a list of problems.
     *
     * @param replyId unique identifier of the reply
     * @param problems list of problems that caused the rejection
     * @return confirmation of the rejection
     * @throws RestApiException if a technical error occurred
     */
    String rejectReply(UUID replyId, List<Problem> problems) throws RestApiException;

    /**
     * Gets the submission state.
     *
     * @param submission the submission to check
     * @return the current state of the submission
     */
    Status getSubmitState(Submission submission);

    /**
     * Scans a byte array for viruses using the configured virus scanner.
     *
     * @param data the byte array to scan for viruses
     * @return VirusScanResult containing the scan results
     * @throws VirusScanException if the scan fails or the virus scanner is unavailable
     */
    VirusScanResult scanBytesForViruses(byte[] data) throws VirusScanException;

    /**
     * Scans an input stream for viruses using the configured virus scanner.
     *
     * @param inputStream the input stream to scan for viruses
     * @return VirusScanResult containing the scan results
     * @throws VirusScanException if the scan fails or the virus scanner is unavailable
     */
    VirusScanResult scanStreamForViruses(InputStream inputStream) throws VirusScanException;

    /**
     * Scans a file for viruses using the configured virus scanner.
     *
     * @param filePath the path to the file to scan for viruses
     * @return VirusScanResult containing the scan results
     * @throws VirusScanException if the scan fails, the file doesn't exist, or the virus scanner is
     *     unavailable
     */
    VirusScanResult scanFileForViruses(Path filePath) throws VirusScanException;
}
