/*
 * Decompiled with CFR 0.152.
 */
package dev.fitko.fitconnect.jwkvalidator.x5c.ocsp;

import dev.fitko.fitconnect.jwkvalidator.x5c.CertStatus;
import dev.fitko.fitconnect.jwkvalidator.x5c.crl.CRLVerifier;
import dev.fitko.fitconnect.jwkvalidator.x5c.ocsp.OCSPUtils;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SignatureException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import org.bouncycastle.asn1.ocsp.OCSPObjectIdentifiers;
import org.bouncycastle.asn1.ocsp.ResponderID;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.KeyPurposeId;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cert.ocsp.BasicOCSPResp;
import org.bouncycastle.cert.ocsp.OCSPException;
import org.bouncycastle.operator.ContentVerifierProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class OCSPResponseSignatureValidator {
    private static final Logger log = LoggerFactory.getLogger(OCSPResponseSignatureValidator.class);
    private final CRLVerifier crlVerifier;
    private final List<String> validSignatureAlgOIDs;

    OCSPResponseSignatureValidator(CRLVerifier crlVerifier, List<String> ocspResponsSignatureAlgOIDs) {
        this.crlVerifier = crlVerifier;
        this.validSignatureAlgOIDs = this.verifySignatureAlgOIDList(ocspResponsSignatureAlgOIDs);
    }

    private List<String> verifySignatureAlgOIDList(List<String> validSignatureAlgOIDs) {
        if (validSignatureAlgOIDs == null || validSignatureAlgOIDs.isEmpty()) {
            log.warn("Empty list of signature algorithm OIDs for OCSP Response validation.");
            return Collections.emptyList();
        }
        log.debug("List of valid OCSP Response Signature Algorithms: {}", validSignatureAlgOIDs);
        return validSignatureAlgOIDs;
    }

    Optional<BasicOCSPResp> validate(BasicOCSPResp basicResp, String serviceURL, X509Certificate leafCert, X509Certificate intermediateCert) {
        return this.extractDelegationCertFromResponse(basicResp, leafCert, serviceURL).flatMap(delegationCert -> this.delegationCertHasCorrectKeyUsages((X509Certificate)delegationCert, leafCert, serviceURL)).flatMap(delegationCert -> this.delegationCertHasCorrectExtensions((X509Certificate)delegationCert, leafCert, serviceURL)).flatMap(delegationCert -> this.delegationCertSignedCorrectly((X509Certificate)delegationCert, intermediateCert, leafCert, serviceURL)).flatMap(delegationCert -> this.validSignatureAlgorithm(basicResp, (X509Certificate)delegationCert, leafCert, serviceURL)).flatMap(delegationCert -> this.validSignature(basicResp, (X509Certificate)delegationCert, leafCert, serviceURL));
    }

    private Optional<BasicOCSPResp> validSignature(BasicOCSPResp basicOCSPResp, X509Certificate delegationCert, X509Certificate leafCert, String serviceURL) {
        try {
            ContentVerifierProvider contentVerifierProvider = new JcaContentVerifierProviderBuilder().setProvider("BC").build(delegationCert);
            if (!basicOCSPResp.isSignatureValid(contentVerifierProvider)) {
                log.error("Invalid signature in OCSP response for leaf cert {} in url {}", (Object)leafCert.getSerialNumber(), (Object)serviceURL);
                return Optional.empty();
            }
            return Optional.of(basicOCSPResp);
        }
        catch (OperatorCreationException e) {
            log.error("Could not create ContentVerifierProvider for leaf cert {} in url {}", new Object[]{leafCert.getSerialNumber(), serviceURL, e});
        }
        catch (OCSPException e) {
            log.error("Failed verifying signature for leaf cert {} in url {}", new Object[]{leafCert.getSerialNumber(), serviceURL, e});
        }
        return Optional.empty();
    }

    private Optional<X509Certificate> validSignatureAlgorithm(BasicOCSPResp basicOCSPResp, X509Certificate delegationCert, X509Certificate leafCert, String serviceURL) {
        log.debug("[validSignatureAlgorithm] - Signature algorithm {} in response for leaf cert {} in url {}", new Object[]{basicOCSPResp.getSignatureAlgOID(), leafCert.getSerialNumber(), serviceURL});
        String responseSignatureAlgOID = basicOCSPResp.getSignatureAlgOID().getId();
        if (this.validSignatureAlgOIDs.isEmpty()) {
            log.debug("Skipping validation of used signature algorithm in OCSP response");
        } else if (!this.validSignatureAlgOIDs.contains(responseSignatureAlgOID)) {
            log.error("Signature algorithm {} in OCSP response not in list of valid signature algorithm OIDs: {}", (Object)responseSignatureAlgOID, this.validSignatureAlgOIDs);
            return Optional.empty();
        }
        return Optional.of(delegationCert);
    }

    private Optional<X509Certificate> delegationCertHasCorrectExtensions(X509Certificate delegationCert, X509Certificate leafCert, String serviceURL) {
        if (delegationCert.getNonCriticalExtensionOIDs() == null) {
            log.error("Delegation certificate does not have any non critical extensions for leaf cert {} in url {}", (Object)leafCert.getSerialNumber(), (Object)serviceURL);
            return Optional.empty();
        }
        if (delegationCert.getNonCriticalExtensionOIDs().contains(OCSPObjectIdentifiers.id_pkix_ocsp_nocheck.getId())) {
            return Optional.of(delegationCert);
        }
        log.debug("Delegation certificate does not have id-pkix-ocsp-nocheck extension for leaf cert {} in url {}", (Object)leafCert.getSerialNumber(), (Object)serviceURL);
        if (OCSPUtils.readAuthorityInformationAccessBytes(delegationCert).isPresent()) {
            log.warn("Delegation certificate unexpectedly contains OCSP extension {} for leaf cert {} in url {}", new Object[]{Extension.authorityInfoAccess, leafCert.getSerialNumber(), serviceURL});
        }
        if (!delegationCert.getNonCriticalExtensionOIDs().contains(Extension.cRLDistributionPoints.toString())) {
            log.error("Delegation certificate does not have the required CRL Distribution Points extension {} for leaf cert {} in url {}. Signer certificate cannot be verified.", new Object[]{Extension.cRLDistributionPoints, leafCert.getSerialNumber(), serviceURL});
            return Optional.empty();
        }
        return this.validateOCSPSignerCertificate(delegationCert, leafCert, serviceURL);
    }

    private Optional<X509Certificate> validateOCSPSignerCertificate(X509Certificate delegationCert, X509Certificate leafCert, String serviceURL) {
        CertStatus crlStatus = this.crlVerifier.checkCertStatus(delegationCert, delegationCert, false);
        if (crlStatus != CertStatus.VALID) {
            log.error("Delegation certificate {} for leaf cert {} in url {} returned status {} for its CRL check.", new Object[]{delegationCert.getSerialNumber(), leafCert.getSerialNumber(), serviceURL, crlStatus});
            return Optional.empty();
        }
        return Optional.of(delegationCert);
    }

    private Optional<X509Certificate> delegationCertHasCorrectKeyUsages(X509Certificate delegationCert, X509Certificate leafCert, String serviceURL) {
        try {
            if (delegationCert.getExtendedKeyUsage() == null) {
                log.error("Delegation certificate does not have any extended key usages for leaf cert {} in url {}", (Object)leafCert.getSerialNumber(), (Object)serviceURL);
                return Optional.empty();
            }
            if (!delegationCert.getExtendedKeyUsage().contains(KeyPurposeId.id_kp_OCSPSigning.getId())) {
                log.error("Delegation certificate does not have id-kp-OCSPSigning extension for leaf cert {} in url {}", (Object)leafCert.getSerialNumber(), (Object)serviceURL);
                return Optional.empty();
            }
            return Optional.of(delegationCert);
        }
        catch (CertificateParsingException e) {
            log.error("Cannot extract extended key usage form delegation cert for leaf cert {} in url {}", new Object[]{leafCert.getSerialNumber(), serviceURL, e});
            return Optional.empty();
        }
    }

    private Optional<X509Certificate> delegationCertSignedCorrectly(X509Certificate delegationCert, X509Certificate intermediateCert, X509Certificate leafCert, String serviceURL) {
        try {
            log.debug("[delegationCertSignedCorrectly] - Signature algorithm {} ({}) in delegation cert {} for leaf cert {} in url {}", new Object[]{delegationCert.getSigAlgOID(), delegationCert.getSigAlgName(), delegationCert.getSerialNumber(), leafCert.getSerialNumber(), serviceURL});
            delegationCert.verify(intermediateCert.getPublicKey());
            return Optional.of(delegationCert);
        }
        catch (CertificateException e) {
            log.error("Delegation cert has wrong encoding for leaf cert {} in url {}", new Object[]{leafCert.getSerialNumber(), serviceURL, e});
        }
        catch (NoSuchAlgorithmException e) {
            log.error("Signature algorithm in delegation cert is not supported for leaf cert {} in url {}", new Object[]{leafCert.getSerialNumber(), serviceURL, e});
        }
        catch (InvalidKeyException e) {
            log.error("Incorrect key in delegation cert check with leaf cert {} in url {}", new Object[]{leafCert.getSerialNumber(), serviceURL, e});
        }
        catch (NoSuchProviderException e) {
            log.error("No provider found for signature check with leaf cert {} in url {}", new Object[]{leafCert.getSerialNumber(), serviceURL, e});
        }
        catch (SignatureException e) {
            log.error("Incorrect signature for delegation cert with leaf cert {} in url {}", new Object[]{leafCert.getSerialNumber(), serviceURL, e});
        }
        return Optional.empty();
    }

    private Optional<X509Certificate> extractDelegationCertFromResponse(BasicOCSPResp response, X509Certificate leafCert, String serviceURL) {
        JcaX509CertificateConverter certificateConverter = new JcaX509CertificateConverter();
        ResponderID responderID = response.getResponderId().toASN1Primitive();
        X509Certificate signatureCert = this.extractSignatureCertFromResponseByKeyHash(response, leafCert, serviceURL, responderID, certificateConverter);
        if (signatureCert != null) {
            return Optional.of(signatureCert);
        }
        signatureCert = this.extractSignatureCertFromResponseByName(response, leafCert, serviceURL, responderID, certificateConverter);
        if (signatureCert != null) {
            return Optional.of(signatureCert);
        }
        log.error("Cannot extract signature cert from response for leaf cert {} in url {}", (Object)leafCert.getSerialNumber(), (Object)serviceURL);
        return Optional.empty();
    }

    private X509Certificate extractSignatureCertFromResponseByName(BasicOCSPResp response, X509Certificate leafCert, String serviceURL, ResponderID responderID, JcaX509CertificateConverter certificateConverter) {
        X509CertificateHolder[] certHolders;
        X500Name x500Name = responderID.getName();
        if (x500Name == null) {
            return null;
        }
        for (X509CertificateHolder certHolder : certHolders = response.getCerts()) {
            if (!x500Name.equals((Object)certHolder.getSubject())) continue;
            try {
                return certificateConverter.getCertificate(certHolder);
            }
            catch (CertificateException e) {
                log.warn("Failed converting signature cert based on name for leaf cert {} in url {}", new Object[]{leafCert.getSerialNumber(), serviceURL, e});
            }
        }
        return null;
    }

    private X509Certificate extractSignatureCertFromResponseByKeyHash(BasicOCSPResp response, X509Certificate leafCert, String serviceURL, ResponderID responderID, JcaX509CertificateConverter certificateConverter) {
        X509CertificateHolder[] certHolders;
        byte[] keyHash = responderID.getKeyHash();
        if (keyHash == null) {
            return null;
        }
        for (X509CertificateHolder certHolder : certHolders = response.getCerts()) {
            byte[] digest = this.getKeyHashFromCertHolder(certHolder);
            if (!Arrays.equals(keyHash, digest)) continue;
            try {
                return certificateConverter.getCertificate(certHolder);
            }
            catch (CertificateException e) {
                log.warn("Failed converting signature cert based on key hash for leaf cert {} in url {}", new Object[]{leafCert.getSerialNumber(), serviceURL, e});
            }
        }
        return null;
    }

    private byte[] getKeyHashFromCertHolder(X509CertificateHolder certHolder) {
        try {
            SubjectPublicKeyInfo info = certHolder.getSubjectPublicKeyInfo();
            return MessageDigest.getInstance("SHA-1").digest(info.getPublicKeyData().getBytes());
        }
        catch (NoSuchAlgorithmException e) {
            log.error("No algorithm of type SHA-1", (Throwable)e);
            return null;
        }
    }
}

