/*
 * Decompiled with CFR 0.152.
 */
package dev.fitko.fitconnect.jwkvalidator.x5c.crl;

import dev.fitko.fitconnect.jwkvalidator.x5c.crl.CRLDownloadRequest;
import java.net.Proxy;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SignatureException;
import java.security.cert.CRLException;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import org.bouncycastle.asn1.x509.DistributionPoint;
import org.bouncycastle.asn1.x509.Extension;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class CRLDownloadManager {
    private static final Logger log = LoggerFactory.getLogger(CRLDownloadManager.class);
    private final Proxy proxy;
    private final List<String> validSignatureAlgOIDs;

    CRLDownloadManager(Proxy proxy, List<String> crlSignatureAlgOIDs) {
        this.proxy = proxy == null ? Proxy.NO_PROXY : proxy;
        this.validSignatureAlgOIDs = this.verifySignatureAlgOIDList(crlSignatureAlgOIDs);
    }

    private List<String> verifySignatureAlgOIDList(List<String> validSignatureAlgOIDs) {
        if (validSignatureAlgOIDs == null || validSignatureAlgOIDs.isEmpty()) {
            log.warn("Empty list of signature algorithm OIDs for CRL validation.");
            return Collections.emptyList();
        }
        log.debug("List of valid CRL Signature Algorithms: {}", validSignatureAlgOIDs);
        return validSignatureAlgOIDs;
    }

    Optional<X509CRL> downloadAndValidateCRL(DistributionPoint dp, X509Certificate cert, X509Certificate signingCert, String crlURL, boolean ifNoCrlSignKeyUsageThanInvalid) {
        Optional<X509CRL> x509CRL = this.downloadCRL(cert, crlURL);
        if (x509CRL.isEmpty()) {
            return Optional.empty();
        }
        return this.validateCRL(x509CRL.get(), dp, cert, signingCert, ifNoCrlSignKeyUsageThanInvalid);
    }

    private Optional<X509CRL> downloadCRL(X509Certificate cert, String crlURL) {
        return new CRLDownloadRequest(this.proxy).downloadResponse(cert, crlURL);
    }

    private Optional<X509CRL> validateCRL(X509CRL x509CRL, DistributionPoint dp, X509Certificate cert, X509Certificate signingCert, boolean ifNoCrlSignKeyUsageThanInvalid) {
        return this.validSignatureAlgorithm(x509CRL, cert, signingCert).flatMap(crl -> this.x509CRLIsValid(x509CRL, cert, signingCert, ifNoCrlSignKeyUsageThanInvalid)).flatMap(crl -> this.validCRLIssuerInDistributionPoint((X509CRL)crl, dp, cert)).flatMap(crl -> this.validIssuingDistributionPointInCRL((X509CRL)crl, cert)).flatMap(crl -> this.validNextUpdateValue((X509CRL)crl, cert)).flatMap(crl -> this.validateDeltaCRLIndicator((X509CRL)crl, cert));
    }

    private Optional<X509CRL> validSignatureAlgorithm(X509CRL x509CRL, X509Certificate cert, X509Certificate signingCert) {
        log.debug("[validSignatureAlgorithm] - Signature algorithm {} ({}) in x509CRL and {} ({}) in signingCert {} for cert {}", new Object[]{x509CRL.getSigAlgOID(), x509CRL.getSigAlgName(), signingCert.getSigAlgOID(), signingCert.getSigAlgName(), signingCert.getSerialNumber(), cert.getSerialNumber()});
        String responseSignatureAlgOID = x509CRL.getSigAlgOID();
        if (this.validSignatureAlgOIDs.isEmpty()) {
            log.debug("Skipping validation of used signature algorithm in CRL");
        } else if (!this.validSignatureAlgOIDs.contains(responseSignatureAlgOID)) {
            log.error("Signature algorithm {} in CRL response not in list of valid signature algorithm OIDs: {}", (Object)responseSignatureAlgOID, this.validSignatureAlgOIDs);
            return Optional.empty();
        }
        return Optional.of(x509CRL);
    }

    private Optional<X509CRL> x509CRLIsValid(X509CRL x509CRL, X509Certificate cert, X509Certificate signingCert, boolean ifNoCrlSignKeyUsageThanInvalid) {
        try {
            if (signingCert.getKeyUsage() != null && !signingCert.getKeyUsage()[6]) {
                log.error("CRLSign key usage not specified for cert {} and signing cert {}", (Object)cert.getSerialNumber(), (Object)signingCert.getSerialNumber());
                return ifNoCrlSignKeyUsageThanInvalid ? Optional.empty() : Optional.of(x509CRL);
            }
            x509CRL.verify(signingCert.getPublicKey());
            return Optional.of(x509CRL);
        }
        catch (CRLException e) {
            log.error("X509CRL has wrong encoding for cert {} and signging cert {}", new Object[]{cert.getSerialNumber(), signingCert.getSerialNumber(), e});
        }
        catch (NoSuchAlgorithmException e) {
            log.error("Signature algorithm in X509CRL is not supported for cert {} and signing cert {}", new Object[]{cert.getSerialNumber(), signingCert.getSerialNumber(), e});
        }
        catch (InvalidKeyException e) {
            log.error("Incorrect key in X509CRL check with cert {} and signing cert {}", new Object[]{cert.getSerialNumber(), signingCert.getSerialNumber(), e});
        }
        catch (NoSuchProviderException e) {
            log.error("No provider found for X509CRL check with cert {} and signing cert {}", new Object[]{cert.getSerialNumber(), signingCert.getSerialNumber(), e});
        }
        catch (SignatureException e) {
            log.error("Incorrect signature for X509CRL check with cert {} and signing cert {}", new Object[]{cert.getSerialNumber(), signingCert.getSerialNumber(), e});
        }
        return Optional.of(x509CRL);
    }

    private Optional<X509CRL> validateDeltaCRLIndicator(X509CRL x509CRL, X509Certificate cert) {
        byte[] idp = x509CRL.getExtensionValue(Extension.deltaCRLIndicator.getId());
        if (idp != null) {
            log.warn("CRL has a delta indicator for cert {}", (Object)cert.getSerialNumber());
            return Optional.empty();
        }
        return Optional.of(x509CRL);
    }

    private Optional<X509CRL> validNextUpdateValue(X509CRL x509CRL, X509Certificate cert) {
        Date curDate = Calendar.getInstance().getTime();
        Date thisUpdate = x509CRL.getThisUpdate();
        if (thisUpdate == null) {
            log.warn("CRL contains no 'thisUpdate' field for cert {}", (Object)cert.getSerialNumber());
            return Optional.empty();
        }
        Date nextUpdate = x509CRL.getNextUpdate();
        if (nextUpdate == null) {
            log.warn("CRL contains no 'nextUpdate' field for cert {}", (Object)cert.getSerialNumber());
            return Optional.empty();
        }
        if (curDate.compareTo(thisUpdate) < 0) {
            log.warn("Current date {} < thisUpdate {} field for cert {}", new Object[]{curDate, thisUpdate, cert});
            return Optional.empty();
        }
        if (curDate.compareTo(nextUpdate) > 0) {
            log.warn("Current date {} > nextUpdate {} field for cert {}", new Object[]{curDate, thisUpdate, cert.getSerialNumber()});
            return Optional.empty();
        }
        return Optional.of(x509CRL);
    }

    private Optional<X509CRL> validIssuingDistributionPointInCRL(X509CRL x509CRL, X509Certificate cert) {
        byte[] idp = x509CRL.getExtensionValue(Extension.issuingDistributionPoint.getId());
        if (idp != null) {
            log.warn("CRL has suddenly an issuing distribution point for cert {}", (Object)cert.getSerialNumber());
            return Optional.empty();
        }
        return Optional.of(x509CRL);
    }

    private Optional<X509CRL> validCRLIssuerInDistributionPoint(X509CRL x509CRL, DistributionPoint dp, X509Certificate cert) {
        if (dp.getCRLIssuer() != null) {
            log.warn("Distribution Point has suddenly an CRL issuer for cert {}", (Object)cert.getSerialNumber());
            return Optional.empty();
        }
        boolean principalsEqual = x509CRL.getIssuerX500Principal().equals(cert.getIssuerX500Principal());
        return principalsEqual ? Optional.of(x509CRL) : Optional.empty();
    }
}

