/*
 * Decompiled with CFR 0.152.
 */
package dev.fitko.fitconnect.jwkvalidator.x5c;

import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.RSAKey;
import com.nimbusds.jose.util.Base64;
import dev.fitko.fitconnect.jwkvalidator.exceptions.JWKValidationException;
import dev.fitko.fitconnect.jwkvalidator.exceptions.LogLevel;
import dev.fitko.fitconnect.jwkvalidator.x5c.CertStatus;
import dev.fitko.fitconnect.jwkvalidator.x5c.X5CUtils;
import dev.fitko.fitconnect.jwkvalidator.x5c.crl.CRLVerifier;
import dev.fitko.fitconnect.jwkvalidator.x5c.ocsp.OCSPVerifier;
import java.net.Proxy;
import java.security.InvalidAlgorithmParameterException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.CertPathValidatorException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateFactory;
import java.security.cert.PKIXParameters;
import java.security.cert.TrustAnchor;
import java.security.cert.X509Certificate;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class X5CValidator {
    private static final Logger log = LoggerFactory.getLogger(X5CValidator.class);
    private static final int CERTIFICATE_CHAIN_LENGTH = 3;
    private final Set<TrustAnchor> trustAnchors;
    private final Date defaultValidationDate;
    private final OCSPVerifier ocspVerifier;
    private final CRLVerifier crlVerifier;
    private final LogLevel logLevel;

    private X5CValidator(Proxy proxy, Set<TrustAnchor> trustAnchors, Date defaultValidationDate, List<String> ocspResponsSignatureAlgOIDs, List<String> crlSignatureAlgOIDs, LogLevel logLevel) {
        this.trustAnchors = trustAnchors;
        this.defaultValidationDate = defaultValidationDate;
        this.crlVerifier = new CRLVerifier(proxy, crlSignatureAlgOIDs);
        this.ocspVerifier = new OCSPVerifier(proxy, this.crlVerifier, ocspResponsSignatureAlgOIDs);
        this.logLevel = logLevel;
    }

    public static X5CValidator of(Proxy proxy, Set<TrustAnchor> trustAnchors, Date defaultValidationDate, List<String> ocspResponsSignatureAlgOIDs, List<String> crlSignatureAlgOIDs, LogLevel logLevel) {
        return new X5CValidator(proxy, trustAnchors, defaultValidationDate, ocspResponsSignatureAlgOIDs, crlSignatureAlgOIDs, logLevel);
    }

    public void validate(RSAKey jwk) throws JWKValidationException {
        this.validate(jwk, this.defaultValidationDate);
    }

    public void validate(RSAKey jwk, Date validationDate) throws JWKValidationException {
        this.hasCorrectAmountOfCertificates(jwk, jwk.getX509CertChain());
        this.hasNoNullCertificates(jwk, jwk.getX509CertChain());
        List x5c = jwk.getParsedX509CertChain();
        this.hasValidCertPath(validationDate, jwk, x5c);
        if (!X5CUtils.moreThan2HoursBeforeNow(validationDate)) {
            this.hasValidOCSPResponse(jwk, (X509Certificate)x5c.get(0), (X509Certificate)x5c.get(1));
        }
        this.hasValidCRLResponse(jwk, (X509Certificate)x5c.get(1), (X509Certificate)x5c.get(2));
        this.hasValidCRLResponse(jwk, (X509Certificate)x5c.get(2), (X509Certificate)x5c.get(2));
    }

    private void hasCorrectAmountOfCertificates(RSAKey jwk, Collection<Base64> x5c) throws JWKValidationException {
        if (x5c.size() != 3) {
            throw JWKValidationException.build((JWK)jwk, log, this.logLevel, "JWK with id {} has an invalid amount of certificates. Is {} but should be {}.", jwk.getKeyID(), x5c.size(), 3);
        }
    }

    private void hasNoNullCertificates(RSAKey jwk, List<Base64> x5c) throws JWKValidationException {
        for (int i = 0; i < 3; ++i) {
            if (x5c.get(i) != null) continue;
            throw JWKValidationException.build((JWK)jwk, log, this.logLevel, "JWK with id {} has a null certificate at position {}", jwk.getKeyID(), i);
        }
    }

    private void hasValidCertPath(Date validationDate, RSAKey jwk, List<X509Certificate> x5c) throws JWKValidationException {
        CertificateFactory cf = this.getX509CertificateFactory(jwk);
        CertPath path = this.generateCertPath(cf, jwk, x5c);
        PKIXParameters pkixParameters = this.buildPKIXParameters(validationDate, jwk);
        CertPathValidator validator = this.getPKIXCertPathValidator(jwk);
        this.validate(validator, path, pkixParameters, jwk);
    }

    private void hasValidOCSPResponse(RSAKey jwk, X509Certificate peerCert, X509Certificate issuerCert) throws JWKValidationException {
        log.debug("[hasValidOCSPResponse] - Signing algorithm {} ({}) in peerCert {} and {} ({}) in issuerCert {} for jwk {}", new Object[]{peerCert.getSigAlgOID(), peerCert.getSigAlgName(), peerCert.getSerialNumber(), issuerCert.getSigAlgOID(), issuerCert.getSigAlgName(), issuerCert.getSerialNumber(), jwk.getKeyID()});
        CertStatus certStatus = this.ocspVerifier.checkCertStatus(peerCert, issuerCert);
        if (certStatus != CertStatus.VALID) {
            throw JWKValidationException.build((JWK)jwk, log, this.logLevel, "JWK with id {} returned certificate status {} for OCSP check", new Object[]{jwk.getKeyID(), certStatus});
        }
    }

    private void hasValidCRLResponse(RSAKey jwk, X509Certificate cert, X509Certificate signingCert) throws JWKValidationException {
        log.debug("[hasValidCRLResponse] - Signing algorithm {} ({}) in cert {} and {} ({}) in signingCert {} for jwk {}", new Object[]{cert.getSigAlgOID(), cert.getSigAlgName(), cert.getSerialNumber(), signingCert.getSigAlgOID(), signingCert.getSigAlgName(), signingCert.getSerialNumber(), jwk.getKeyID()});
        CertStatus certStatus = this.crlVerifier.checkCertStatus(cert, signingCert);
        if (certStatus != CertStatus.VALID) {
            throw JWKValidationException.build((JWK)jwk, log, this.logLevel, "JWK with id {} returned certificate status {} for CRL check of certificate {} with signing cert {}", new Object[]{jwk.getKeyID(), certStatus, cert.getSerialNumber(), signingCert.getSerialNumber()});
        }
    }

    private CertificateFactory getX509CertificateFactory(RSAKey jwk) throws JWKValidationException {
        try {
            return CertificateFactory.getInstance("X.509");
        }
        catch (CertificateException e) {
            throw JWKValidationException.build((JWK)jwk, log, this.logLevel, "X.509 CertificateFactory not supported", e, new Object[0]);
        }
    }

    private CertPath generateCertPath(CertificateFactory cf, RSAKey jwk, List<X509Certificate> x5c) throws JWKValidationException {
        try {
            return cf.generateCertPath(x5c);
        }
        catch (CertificateException e) {
            throw JWKValidationException.build((JWK)jwk, log, this.logLevel, "Failed generating cert path from x5c", e, new Object[0]);
        }
    }

    private PKIXParameters buildPKIXParameters(Date validationDate, RSAKey jwk) throws JWKValidationException {
        try {
            PKIXParameters pkixParameters = new PKIXParameters(this.trustAnchors);
            pkixParameters.setRevocationEnabled(false);
            pkixParameters.setDate(validationDate);
            return pkixParameters;
        }
        catch (InvalidAlgorithmParameterException e) {
            throw JWKValidationException.build((JWK)jwk, log, this.logLevel, "Empty trust anchors", e, new Object[0]);
        }
        catch (NullPointerException e) {
            throw JWKValidationException.build((JWK)jwk, log, this.logLevel, "TrustAnchors are null", e, new Object[0]);
        }
        catch (ClassCastException e) {
            throw JWKValidationException.build((JWK)jwk, log, this.logLevel, "TrustAnchor does not contain TrustAnchors", e, new Object[0]);
        }
    }

    private CertPathValidator getPKIXCertPathValidator(RSAKey jwk) throws JWKValidationException {
        try {
            return CertPathValidator.getInstance("PKIX");
        }
        catch (NoSuchAlgorithmException e) {
            throw JWKValidationException.build((JWK)jwk, log, this.logLevel, "PKIX CertPathValidator not supported", e, new Object[0]);
        }
    }

    private void validate(CertPathValidator validator, CertPath path, PKIXParameters pkixParameters, RSAKey jwk) throws JWKValidationException {
        try {
            validator.validate(path, pkixParameters);
        }
        catch (InvalidAlgorithmParameterException e) {
            throw JWKValidationException.build((JWK)jwk, log, this.logLevel, "CertPathValidator does not support specified algorithm", e, new Object[0]);
        }
        catch (CertPathValidatorException e) {
            this.evaluateCertPathValidatorException(e, jwk);
        }
    }

    private void evaluateCertPathValidatorException(CertPathValidatorException e, RSAKey jwk) throws JWKValidationException {
        Throwable t = this.hasCause(e, CertificateExpiredException.class);
        if (t != null) {
            throw JWKValidationException.build((JWK)jwk, log, this.logLevel, "JWK with id {} failed because certificate expired: {}", e, jwk.getKeyID(), t.getMessage());
        }
        throw JWKValidationException.build((JWK)jwk, log, this.logLevel, "JWK with id {} has invalid certificate chain", e, jwk.getKeyID());
    }

    private Throwable hasCause(Throwable throwable, Class<? extends Throwable> cause) {
        if (throwable == null) {
            return null;
        }
        if (cause.isAssignableFrom(throwable.getClass())) {
            return throwable;
        }
        return this.hasCause(throwable.getCause(), cause);
    }
}

