/*
 *  Primitives.scala
 *  (Lucre)
 *
 *  Copyright (c) 2009-2019 Hanns Holger Rutz. All rights reserved.
 *
 *  This software is published under the GNU Lesser General Public License v2.1+
 *
 *
 *  For further information, please contact Hanns Holger Rutz at
 *  contact@sciss.de
 */

package de.sciss.lucre.expr

import de.sciss.lucre.event.Targets
import de.sciss.lucre.expr
import de.sciss.lucre.stm.Sys
import de.sciss.serial.{ImmutableSerializer, Serializer}
import de.sciss.span.{Span, SpanLike}

import scala.collection.immutable.{IndexedSeq => Vec}

trait IntObj      [S <: Sys[S]] extends Expr[S, Int        ]
trait LongObj     [S <: Sys[S]] extends Expr[S, Long       ]
trait DoubleObj   [S <: Sys[S]] extends Expr[S, Double     ]
trait BooleanObj  [S <: Sys[S]] extends Expr[S, Boolean    ]
trait StringObj   [S <: Sys[S]] extends Expr[S, String     ]
trait SpanLikeObj [S <: Sys[S]] extends Expr[S, SpanLike   ]
trait SpanObj     [S <: Sys[S]] extends Expr[S, Span       ]
trait IntVector   [S <: Sys[S]] extends Expr[S, Vec[Int   ]]
trait DoubleVector[S <: Sys[S]] extends Expr[S, Vec[Double]]

object IntObj extends impl.ExprTypeImpl[Int, IntObj] {
  import expr.{IntObj => Repr}

  final val typeId = 2
  final val valueSerializer = Serializer.Int

  override def toString = "IntObj"

  protected def mkConst[S <: Sys[S]](id: S#Id, value: A)(implicit tx: S#Tx): Const[S] =
    new _Const[S](id, value)

  protected def mkVar[S <: Sys[S]](targets: Targets[S], vr: S#Var[_Ex[S]], connect: Boolean)
                                  (implicit tx: S#Tx): Var[S] = {
    val res = new _Var[S](targets, vr)
    if (connect) res.connect()
    res
  }

  private[this] final class _Const[S <: Sys[S]](val id: S#Id, val constValue: A)
    extends ConstImpl[S] with Repr[S]

  private[this] final class _Var[S <: Sys[S]](val targets: Targets[S], val ref: S#Var[_Ex[S]])
    extends VarImpl[S] with Repr[S]
}

object LongObj extends impl.ExprTypeImpl[Long, LongObj] {
  import expr.{LongObj => Repr}

  final val typeId = 3
  final val valueSerializer = Serializer.Long

  override def toString = "LongObj"

  protected def mkConst[S <: Sys[S]](id: S#Id, value: A)(implicit tx: S#Tx): Const[S] =
    new _Const[S](id, value)

  protected def mkVar[S <: Sys[S]](targets: Targets[S], vr: S#Var[_Ex[S]], connect: Boolean)
                                  (implicit tx: S#Tx): Var[S] = {
    val res = new _Var[S](targets, vr)
    if (connect) res.connect()
    res
  }

  private[this] final class _Const[S <: Sys[S]](val id: S#Id, val constValue: A)
    extends ConstImpl[S] with Repr[S]

  private[this] final class _Var[S <: Sys[S]](val targets: Targets[S], val ref: S#Var[_Ex[S]])
    extends VarImpl[S] with Repr[S]
}

object DoubleObj extends impl.ExprTypeImpl[Double, DoubleObj] {
  import expr.{DoubleObj => Repr}

  final val typeId = 5
  final val valueSerializer = Serializer.Double

  override def toString = "DoubleObj"

  protected def mkConst[S <: Sys[S]](id: S#Id, value: A)(implicit tx: S#Tx): Const[S] =
    new _Const[S](id, value)

  protected def mkVar[S <: Sys[S]](targets: Targets[S], vr: S#Var[_Ex[S]], connect: Boolean)
                                  (implicit tx: S#Tx): Var[S] = {
    val res = new _Var[S](targets, vr)
    if (connect) res.connect()
    res
  }

  private[this] final class _Const[S <: Sys[S]](val id: S#Id, val constValue: A)
    extends ConstImpl[S] with Repr[S]

  private[this] final class _Var[S <: Sys[S]](val targets: Targets[S], val ref: S#Var[_Ex[S]])
    extends VarImpl[S] with Repr[S]
}

object BooleanObj extends impl.ExprTypeImpl[Boolean, BooleanObj] {
  import expr.{BooleanObj => Repr}

  final val typeId = 6
  final val valueSerializer = Serializer.Boolean

  override def toString = "BooleanObj"

  protected def mkConst[S <: Sys[S]](id: S#Id, value: A)(implicit tx: S#Tx): Const[S] =
    new _Const[S](id, value)

  protected def mkVar[S <: Sys[S]](targets: Targets[S], vr: S#Var[_Ex[S]], connect: Boolean)
                                  (implicit tx: S#Tx): Var[S] = {
    val res = new _Var[S](targets, vr)
    if (connect) res.connect()
    res
  }

  private[this] final class _Const[S <: Sys[S]](val id: S#Id, val constValue: A)
    extends ConstImpl[S] with Repr[S]

  private[this] final class _Var[S <: Sys[S]](val targets: Targets[S], val ref: S#Var[_Ex[S]])
    extends VarImpl[S] with Repr[S]
}

object StringObj extends impl.ExprTypeImpl[String, StringObj] {
  import expr.{StringObj => Repr}

  final val typeId = 8
  final val valueSerializer = Serializer.String

  override def toString = "StringObj"

  protected def mkConst[S <: Sys[S]](id: S#Id, value: A)(implicit tx: S#Tx): Const[S] =
    new _Const[S](id, value)

  protected def mkVar[S <: Sys[S]](targets: Targets[S], vr: S#Var[_Ex[S]], connect: Boolean)
                                  (implicit tx: S#Tx): Var[S] = {
    val res = new _Var[S](targets, vr)
    if (connect) res.connect()
    res
  }

  private[this] final class _Const[S <: Sys[S]](val id: S#Id, val constValue: A)
    extends ConstImpl[S] with Repr[S]

  private[this] final class _Var[S <: Sys[S]](val targets: Targets[S], val ref: S#Var[_Ex[S]])
    extends VarImpl[S] with Repr[S]
}

object SpanLikeObj extends impl.ExprTypeImpl[SpanLike, SpanLikeObj] {
  import expr.{SpanLikeObj => Repr}

  final val typeId = 9
  final val valueSerializer: ImmutableSerializer[SpanLike] = SpanLike.serializer

  override def toString = "SpanLikeObj"

  protected def mkConst[S <: Sys[S]](id: S#Id, value: A)(implicit tx: S#Tx): Const[S] =
    new _Const[S](id, value)

  protected def mkVar[S <: Sys[S]](targets: Targets[S], vr: S#Var[_Ex[S]], connect: Boolean)
                                  (implicit tx: S#Tx): Var[S] = {
    val res = new _Var[S](targets, vr)
    if (connect) res.connect()
    res
  }

  private[this] final class _Const[S <: Sys[S]](val id: S#Id, val constValue: A)
    extends ConstImpl[S] with Repr[S]

  private[this] final class _Var[S <: Sys[S]](val targets: Targets[S], val ref: S#Var[_Ex[S]])
    extends VarImpl[S] with Repr[S]
}

object SpanObj extends impl.ExprTypeImpl[Span, SpanObj] {
  import expr.{SpanObj => Repr}

  final val typeId = 10
  final val valueSerializer: ImmutableSerializer[Span] = Span.serializer

  override def toString = "SpanObj"

  protected def mkConst[S <: Sys[S]](id: S#Id, value: A)(implicit tx: S#Tx): Const[S] =
    new _Const[S](id, value)

  protected def mkVar[S <: Sys[S]](targets: Targets[S], vr: S#Var[_Ex[S]], connect: Boolean)
                                  (implicit tx: S#Tx): Var[S] = {
    val res = new _Var[S](targets, vr)
    if (connect) res.connect()
    res
  }

  private[this] final class _Const[S <: Sys[S]](val id: S#Id, val constValue: A)
    extends ConstImpl[S] with Repr[S]

  private[this] final class _Var[S <: Sys[S]](val targets: Targets[S], val ref: S#Var[_Ex[S]])
    extends VarImpl[S] with Repr[S]
}

object IntVector extends impl.ExprTypeImpl[Vec[Int], IntVector] {
  import expr.{IntVector => Repr}

  final val typeId = 0x2002 //  0x2000 | IntObj.typeId
  final val valueSerializer: ImmutableSerializer[Vec[Int]] = ImmutableSerializer.indexedSeq

  override def toString = "IntVector"

  protected def mkConst[S <: Sys[S]](id: S#Id, value: A)(implicit tx: S#Tx): Const[S] =
    new _Const[S](id, value)

  protected def mkVar[S <: Sys[S]](targets: Targets[S], vr: S#Var[_Ex[S]], connect: Boolean)
                                  (implicit tx: S#Tx): Var[S] = {
    val res = new _Var[S](targets, vr)
    if (connect) res.connect()
    res
  }

  private[this] final class _Const[S <: Sys[S]](val id: S#Id, val constValue: A)
    extends ConstImpl[S] with Repr[S]

  private[this] final class _Var[S <: Sys[S]](val targets: Targets[S], val ref: S#Var[_Ex[S]])
    extends VarImpl[S] with Repr[S]
}

object DoubleVector extends impl.ExprTypeImpl[Vec[Double], DoubleVector] {
  import expr.{DoubleVector => Repr}

  final val typeId = 0x2005 //  0x2000 | DoubleObj.typeId
  final val valueSerializer: ImmutableSerializer[Vec[Double]] = ImmutableSerializer.indexedSeq

  override def toString = "DoubleVector"

  protected def mkConst[S <: Sys[S]](id: S#Id, value: A)(implicit tx: S#Tx): Const[S] =
    new _Const[S](id, value)

  protected def mkVar[S <: Sys[S]](targets: Targets[S], vr: S#Var[_Ex[S]], connect: Boolean)
                                  (implicit tx: S#Tx): Var[S] = {
    val res = new _Var[S](targets, vr)
    if (connect) res.connect()
    res
  }

  private[this] final class _Const[S <: Sys[S]](val id: S#Id, val constValue: A)
    extends ConstImpl[S] with Repr[S]

  private[this] final class _Var[S <: Sys[S]](val targets: Targets[S], val ref: S#Var[_Ex[S]])
    extends VarImpl[S] with Repr[S]
}
