/*
 * Copyright (C) 2013 Sergej Shafarenka, halfbit.de
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file kt in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.hamm.pinnedsectionlistview;

import android.content.Context;
import android.content.res.TypedArray;
import android.database.DataSetObserver;
import android.graphics.Canvas;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.os.Parcelable;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.SoundEffectConstants;
import android.view.View;
import android.view.ViewConfiguration;
import android.view.accessibility.AccessibilityEvent;
import android.widget.AbsListView;
import android.widget.HeaderViewListAdapter;
import android.widget.ListAdapter;
import android.widget.ListView;
import android.widget.SectionIndexer;

import org.jetbrains.annotations.NotNull;

/**
 * ListView, which is capable to pin section views at its top while the rest is still scrolled.
 */
public class PinnedSectionListView extends ListView {
	private final Rect touchRect = new Rect();
	private final PointF touchPoint = new PointF();

	//-- class fields
	/**
	 * Default change observer.
	 */
	private final DataSetObserver dataSetObserver = new DataSetObserver() {
		@Override
		public void onChanged() {
			recreatePinnedShadow();
		}

		@Override
		public void onInvalidated() {
			recreatePinnedShadow();
		}
	};
	/**
	 * Delegating listener, can be null.
	 */
	OnScrollListener delegateOnScrollListener;
	/**
	 * Scroll listener which does the magic
	 */
	private final OnScrollListener onScrollListener = new OnScrollListener() {
		@Override
		public void onScrollStateChanged(AbsListView view, int scrollState) {
			if (delegateOnScrollListener != null) { // delegate
				delegateOnScrollListener.onScrollStateChanged(view, scrollState);
			}
		}

		@Override
		public void onScroll(AbsListView view, int firstVisibleItem, int visibleItemCount, int totalItemCount) {

			if (delegateOnScrollListener != null) { // delegate
				delegateOnScrollListener.onScroll(view, firstVisibleItem, visibleItemCount, totalItemCount);
			}

			// get expected adapter or fail fast
			ListAdapter adapter = getAdapter();
			if (adapter == null || visibleItemCount == 0) return; // nothing to do

			final boolean isFirstVisibleItemSection =
					isItemViewTypePinned(adapter, adapter.getItemViewType(firstVisibleItem));

			if (isFirstVisibleItemSection) {
				View sectionView = getChildAt(0);
				if (sectionView.getTop() == getPaddingTop()) { // view sticks to the top, no need for pinned shadow
					destroyPinnedShadow();
				} else { // section doesn't stick to the top, make sure we have a pinned shadow
					ensureShadowForPosition(firstVisibleItem, firstVisibleItem, visibleItemCount);
				}

			} else { // section is not at the first visible position
				int sectionPosition = findCurrentSectionPosition(firstVisibleItem);
				if (sectionPosition > -1) { // we have section position
					ensureShadowForPosition(sectionPosition, firstVisibleItem, visibleItemCount);
				} else { // there is no section for the first visible item, destroy shadow
					destroyPinnedShadow();
				}
			}
		}
	};
	/**
	 * Shadow for being recycled, can be null.
	 */
	PinnedSection recycleSection;
	/**
	 * shadow instance with a pinned view, can be null.
	 */
	PinnedSection pinnedSection;
	/**
	 * Pinned view Y-translation. We use it to stick pinned view to the next section.
	 */
	int translateY;
	private int touchSlop;
	private View touchTarget;
	private MotionEvent downEvent;
	// fields used for drawing shadow under a pinned section
	private Drawable shadowDrawable;
	private int shadowHeight;
	private int sectionsDistanceY;

	public PinnedSectionListView(Context context) {
		super(context);
		initView();
		shadowDrawable = context.getResources().getDrawable(R.drawable.default_shadow);
		if (!isInEditMode()) {
			shadowHeight = context.getResources().getDimensionPixelSize(R.dimen.default_shadow_height);
		}
	}

	public PinnedSectionListView(Context context, AttributeSet attrs) {
		super(context, attrs);
		initAttrs(context, attrs);
		initView();
	}

	public PinnedSectionListView(Context context, AttributeSet attrs, int defStyle) {
		super(context, attrs, defStyle);
		initAttrs(context, attrs);
		initView();
	}

	private static boolean isItemViewTypePinned(ListAdapter adapter, int viewType) {
		if (adapter instanceof HeaderViewListAdapter) {
			adapter = ((HeaderViewListAdapter) adapter).getWrappedAdapter();
		}
		return ((PinnedSectionListAdapter) adapter).isItemViewTypePinned(viewType);
	}

	private void initAttrs(Context context, AttributeSet attrs) {
		TypedArray typedArray = context.obtainStyledAttributes(attrs, R.styleable.PinnedSectionListView);
		if (typedArray.hasValue(R.styleable.PinnedSectionListView_shadow)) {
			shadowDrawable = typedArray.getDrawable(R.styleable.PinnedSectionListView_shadow);
		} else {
			shadowDrawable = context.getResources().getDrawable(R.drawable.default_shadow);
		}
		if (typedArray.hasValue(R.styleable.PinnedSectionListView_shadow_height)) {
			shadowHeight = typedArray.getDimensionPixelSize(R.styleable.PinnedSectionListView_shadow_height, 0);
		} else {
			if (!isInEditMode()) {
				shadowHeight = context.getResources().getDimensionPixelSize(R.dimen.default_shadow_height);
			}
		}
		typedArray.recycle();
	}

	private void initView() {
		setOnScrollListener(onScrollListener);
		touchSlop = ViewConfiguration.get(getContext()).getScaledTouchSlop();
	}

	//-- pinned section drawing methods

	/**
	 * Create shadow wrapper with a pinned view for a view at given position
	 */
	void createPinnedShadow(int position) {

		// try to recycle shadow
		PinnedSection pinnedShadow = recycleSection;
		recycleSection = null;

		// create new shadow, if needed
		if (pinnedShadow == null) pinnedShadow = new PinnedSection();
		// request new view using recycled view, if such
		View pinnedView = getAdapter().getView(position, pinnedShadow.view, PinnedSectionListView.this);

		// read layout parameters
		LayoutParams layoutParams = (LayoutParams) pinnedView.getLayoutParams();
		if (layoutParams == null) { // create default layout params
			layoutParams = new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT);
		}

		int heightMode = MeasureSpec.getMode(layoutParams.height);
		int heightSize = MeasureSpec.getSize(layoutParams.height);

		if (heightMode == MeasureSpec.UNSPECIFIED) heightMode = MeasureSpec.EXACTLY;

		int maxHeight = getHeight() - getListPaddingTop() - getListPaddingBottom();
		if (heightSize > maxHeight) heightSize = maxHeight;

		// measure & layout
		int ws = MeasureSpec.makeMeasureSpec(getWidth() - getListPaddingLeft() - getListPaddingRight(),
				MeasureSpec.EXACTLY);
		int hs = MeasureSpec.makeMeasureSpec(heightSize, heightMode);
		pinnedView.measure(ws, hs);
		pinnedView.layout(0, 0, pinnedView.getMeasuredWidth(), pinnedView.getMeasuredHeight());
		translateY = 0;

		// initialize pinned shadow
		pinnedShadow.view = pinnedView;
		pinnedShadow.position = position;
		pinnedShadow.id = getAdapter().getItemId(position);

		// store pinned shadow
		pinnedSection = pinnedShadow;
	}

	/**
	 * Destroy shadow wrapper for currently pinned view
	 */
	void destroyPinnedShadow() {
		if (pinnedSection != null) {
			// keep shadow for being recycled later
			recycleSection = pinnedSection;
			pinnedSection = null;
		}
	}

	/**
	 * Makes sure we have an actual pinned shadow for given position.
	 */
	void ensureShadowForPosition(int sectionPosition, int firstVisibleItem, int visibleItemCount) {
		if (visibleItemCount < 2) { // no need for creating shadow at all, we have a single visible item
			destroyPinnedShadow();
			return;
		}

		if (pinnedSection != null
				&& pinnedSection.position != sectionPosition) { // invalidate shadow, if required
			destroyPinnedShadow();
		}

		if (pinnedSection == null) { // create shadow, if empty
			createPinnedShadow(sectionPosition);
		}

		// align shadow according to next section position, if needed
		int nextPosition = sectionPosition + 1;
		if (nextPosition < getCount()) {
			int nextSectionPosition = findFirstVisibleSectionPosition(nextPosition,
					visibleItemCount - (nextPosition - firstVisibleItem));
			if (nextSectionPosition > -1) {
				View nextSectionView = getChildAt(nextSectionPosition - firstVisibleItem);
				final int bottom = pinnedSection.view.getBottom() + getPaddingTop();
				sectionsDistanceY = nextSectionView.getTop() - bottom;
				if (sectionsDistanceY < 0) {
					// next section overlaps pinned shadow, move it up
					translateY = sectionsDistanceY;
				} else {
					// next section does not overlap with pinned, stick to top
					translateY = 0;
				}
			} else {
				// no other sections are visible, stick to top
				translateY = 0;
				sectionsDistanceY = Integer.MAX_VALUE;
			}
		}

	}

	int findFirstVisibleSectionPosition(int firstVisibleItem, int visibleItemCount) {
		ListAdapter adapter = getAdapter();
		for (int childIndex = 0; childIndex < visibleItemCount; childIndex++) {
			int position = firstVisibleItem + childIndex;
			int viewType = adapter.getItemViewType(position);
			if (isItemViewTypePinned(adapter, viewType)) return position;
		}
		return -1;
	}

	int findCurrentSectionPosition(int fromPosition) {
		ListAdapter adapter = getAdapter();

		if (adapter instanceof SectionIndexer) {
			// try fast way by asking section indexer
			SectionIndexer indexer = (SectionIndexer) adapter;
			int sectionPosition = indexer.getSectionForPosition(fromPosition);
			int itemPosition = indexer.getPositionForSection(sectionPosition);
			int typeView = adapter.getItemViewType(itemPosition);
			if (isItemViewTypePinned(adapter, typeView)) {
				return itemPosition;
			} // else, no luck
		}

		// try slow way by looking through to the next section item above
		for (int position = fromPosition; position >= 0; position--) {
			int viewType = adapter.getItemViewType(position);
			if (isItemViewTypePinned(adapter, viewType)) return position;
		}
		return -1; // no candidate found
	}

	void recreatePinnedShadow() {
		destroyPinnedShadow();
		ListAdapter adapter = getAdapter();
		if (adapter != null && adapter.getCount() > 0) {
			int firstVisiblePosition = getFirstVisiblePosition();
			int sectionPosition = findCurrentSectionPosition(firstVisiblePosition);
			if (sectionPosition == -1) return; // no views to pin, exit
			ensureShadowForPosition(sectionPosition,
					firstVisiblePosition, getLastVisiblePosition() - firstVisiblePosition);
		}
	}

	@Override
	public void setOnScrollListener(OnScrollListener listener) {
		if (listener == onScrollListener) {
			super.setOnScrollListener(listener);
		} else {
			delegateOnScrollListener = listener;
		}
	}

	@Override
	public void onRestoreInstanceState(Parcelable state) {
		super.onRestoreInstanceState(state);
		post(new Runnable() {
			@Override
			public void run() { // restore pinned view after configuration change
				recreatePinnedShadow();
			}
		});
	}

	@Override
	public void setAdapter(ListAdapter adapter) {

		// assert adapter in debug mode
		if (BuildConfig.DEBUG && adapter != null) {
			if (!(adapter instanceof PinnedSectionListAdapter))
				throw new IllegalArgumentException("Does your adapter implement PinnedSectionListAdapter?");
			if (adapter.getViewTypeCount() < 2)
				throw new IllegalArgumentException("Does your adapter handle at least two types" +
						" of views in getViewTypeCount() method: items and sections?");
		}

		// unregister observer at old adapter and register on new one
		ListAdapter oldAdapter = getAdapter();
		if (oldAdapter != null) oldAdapter.unregisterDataSetObserver(dataSetObserver);
		if (adapter != null) adapter.registerDataSetObserver(dataSetObserver);

		// destroy pinned shadow, if new adapter is not same as old one
		if (oldAdapter != adapter) destroyPinnedShadow();

		super.setAdapter(adapter);
	}

	@Override
	protected void onLayout(boolean changed, int l, int t, int r, int b) {
		super.onLayout(changed, l, t, r, b);
		if (pinnedSection != null) {
			int parentWidth = r - l - getPaddingLeft() - getPaddingRight();
			int shadowWidth = pinnedSection.view.getWidth();
			if (parentWidth != shadowWidth) {
				recreatePinnedShadow();
			}
		}
	}

	@Override
	protected void dispatchDraw(@NotNull Canvas canvas) {
		super.dispatchDraw(canvas);

		if (pinnedSection != null) {

			// prepare variables
			int pLeft = getListPaddingLeft();
			int pTop = getListPaddingTop();
			View view = pinnedSection.view;

			// draw child
			canvas.save();

			int clipHeight = view.getHeight() +
					(shadowDrawable == null ? 0 : Math.min(shadowHeight, sectionsDistanceY));
			canvas.clipRect(pLeft, pTop, pLeft + view.getWidth(), pTop + clipHeight);

			canvas.translate(pLeft, pTop + translateY);
			drawChild(canvas, pinnedSection.view, getDrawingTime());

			if (shadowDrawable != null && sectionsDistanceY > 0) {
				shadowDrawable.setBounds(pinnedSection.view.getLeft(),
						pinnedSection.view.getBottom(),
						pinnedSection.view.getRight(),
						pinnedSection.view.getBottom() + shadowHeight);
				shadowDrawable.draw(canvas);
			}

			canvas.restore();
		}
	}

	@Override
	public boolean dispatchTouchEvent(@NotNull MotionEvent ev) {
		final float x = ev.getX();
		final float y = ev.getY();
		final int action = ev.getAction();

		if (action == MotionEvent.ACTION_DOWN
				&& touchTarget == null
				&& pinnedSection != null
				&& isPinnedViewTouched(pinnedSection.view, x, y)) { // create touch target

			// user touched pinned view
			touchTarget = pinnedSection.view;
			touchPoint.x = x;
			touchPoint.y = y;

			// copy down event for eventually be used later
			downEvent = MotionEvent.obtain(ev);
		}

		if (touchTarget != null) {
			if (isPinnedViewTouched(touchTarget, x, y)) { // forward event to pinned view
				touchTarget.dispatchTouchEvent(ev);
			}

			if (action == MotionEvent.ACTION_UP) { // perform onClick on pinned view
				super.dispatchTouchEvent(ev);
				performPinnedItemClick();
				clearTouchTarget();

			} else if (action == MotionEvent.ACTION_CANCEL) { // cancel
				clearTouchTarget();

			} else if (action == MotionEvent.ACTION_MOVE) {
				if (Math.abs(y - touchPoint.y) > touchSlop) {

					// cancel sequence on touch target
					MotionEvent event = MotionEvent.obtain(ev);
					event.setAction(MotionEvent.ACTION_CANCEL);
					touchTarget.dispatchTouchEvent(event);
					event.recycle();

					// provide correct sequence to super class for further handling
					super.dispatchTouchEvent(downEvent);
					super.dispatchTouchEvent(ev);
					clearTouchTarget();

				}
			}

			return true;
		}

		// call super if this was not our pinned view
		return super.dispatchTouchEvent(ev);
	}

	//-- touch handling methods

	private boolean isPinnedViewTouched(View view, float x, float y) {
		view.getHitRect(touchRect);

		// by taping top or bottom padding, the list performs on click on a border item.
		// we don't add top padding here to keep behavior consistent.
		touchRect.top += translateY;

		touchRect.bottom += translateY + getPaddingTop();
		touchRect.left += getPaddingLeft();
		touchRect.right -= getPaddingRight();
		return touchRect.contains((int) x, (int) y);
	}

	private void clearTouchTarget() {
		touchTarget = null;
		if (downEvent != null) {
			downEvent.recycle();
			downEvent = null;
		}
	}

	private boolean performPinnedItemClick() {
		if (pinnedSection == null) return false;

		OnItemClickListener listener = getOnItemClickListener();
		if (listener != null) {
			View view = pinnedSection.view;
			playSoundEffect(SoundEffectConstants.CLICK);
			if (view != null) {
				view.sendAccessibilityEvent(AccessibilityEvent.TYPE_VIEW_CLICKED);
			}
			listener.onItemClick(this, view, pinnedSection.position, pinnedSection.id);
			return true;
		}
		return false;
	}

	/**
	 * List adapter to be implemented for being used with PinnedSectionListView adapter.
	 */
	public static interface PinnedSectionListAdapter extends ListAdapter {
		/**
		 * This method shall return 'true' if views of given type has to be pinned.
		 */
		boolean isItemViewTypePinned(int viewType);
	}

	/**
	 * Wrapper class for pinned section view and its position in the list.
	 */
	static class PinnedSection {
		public View view;
		public int position;
		public long id;
	}

}
