package de.flapdoodle.embed.mongo.commands;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * Immutable implementation of {@link MongoShellArguments}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableMongoShellArguments.builder()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableMongoShellArguments extends MongoShellArguments {
  private final List<String> scriptParameters;
  private final String scriptName;
  private final String dbName;
  private final String password;
  private final String userName;

  private ImmutableMongoShellArguments(
      List<String> scriptParameters,
      String scriptName,
      String dbName,
      String password,
      String userName) {
    this.scriptParameters = scriptParameters;
    this.scriptName = scriptName;
    this.dbName = dbName;
    this.password = password;
    this.userName = userName;
  }

  /**
   * @return The value of the {@code scriptParameters} attribute
   */
  @Override
  public List<String> scriptParameters() {
    return scriptParameters;
  }

  /**
   * @return The value of the {@code scriptName} attribute
   */
  @Override
  public Optional<String> scriptName() {
    return Optional.ofNullable(scriptName);
  }

  /**
   * @return The value of the {@code dbName} attribute
   */
  @Override
  public Optional<String> dbName() {
    return Optional.ofNullable(dbName);
  }

  /**
   * @return The value of the {@code password} attribute
   */
  @Override
  public Optional<String> password() {
    return Optional.ofNullable(password);
  }

  /**
   * @return The value of the {@code userName} attribute
   */
  @Override
  public Optional<String> userName() {
    return Optional.ofNullable(userName);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link MongoShellArguments#scriptParameters() scriptParameters}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableMongoShellArguments withScriptParameters(String... elements) {
    List<String> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableMongoShellArguments(newValue, this.scriptName, this.dbName, this.password, this.userName);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link MongoShellArguments#scriptParameters() scriptParameters}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of scriptParameters elements to set
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableMongoShellArguments withScriptParameters(Iterable<String> elements) {
    if (this.scriptParameters == elements) return this;
    List<String> newValue = createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableMongoShellArguments(newValue, this.scriptName, this.dbName, this.password, this.userName);
  }

  /**
   * Copy the current immutable object by setting a <em>present</em> value for the optional {@link MongoShellArguments#scriptName() scriptName} attribute.
   * @param value The value for scriptName
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableMongoShellArguments withScriptName(String value) {
    String newValue = Objects.requireNonNull(value, "scriptName");
    if (Objects.equals(this.scriptName, newValue)) return this;
    return new ImmutableMongoShellArguments(this.scriptParameters, newValue, this.dbName, this.password, this.userName);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link MongoShellArguments#scriptName() scriptName} attribute.
   * An equality check is used on inner value to prevent copying of the same value by returning {@code this}.
   * @param optional An optional value for scriptName
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableMongoShellArguments withScriptName(Optional<String> optional) {
    String value = optional.orElse(null);
    if (Objects.equals(this.scriptName, value)) return this;
    return new ImmutableMongoShellArguments(this.scriptParameters, value, this.dbName, this.password, this.userName);
  }

  /**
   * Copy the current immutable object by setting a <em>present</em> value for the optional {@link MongoShellArguments#dbName() dbName} attribute.
   * @param value The value for dbName
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableMongoShellArguments withDbName(String value) {
    String newValue = Objects.requireNonNull(value, "dbName");
    if (Objects.equals(this.dbName, newValue)) return this;
    return new ImmutableMongoShellArguments(this.scriptParameters, this.scriptName, newValue, this.password, this.userName);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link MongoShellArguments#dbName() dbName} attribute.
   * An equality check is used on inner value to prevent copying of the same value by returning {@code this}.
   * @param optional An optional value for dbName
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableMongoShellArguments withDbName(Optional<String> optional) {
    String value = optional.orElse(null);
    if (Objects.equals(this.dbName, value)) return this;
    return new ImmutableMongoShellArguments(this.scriptParameters, this.scriptName, value, this.password, this.userName);
  }

  /**
   * Copy the current immutable object by setting a <em>present</em> value for the optional {@link MongoShellArguments#password() password} attribute.
   * @param value The value for password
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableMongoShellArguments withPassword(String value) {
    String newValue = Objects.requireNonNull(value, "password");
    if (Objects.equals(this.password, newValue)) return this;
    return new ImmutableMongoShellArguments(this.scriptParameters, this.scriptName, this.dbName, newValue, this.userName);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link MongoShellArguments#password() password} attribute.
   * An equality check is used on inner value to prevent copying of the same value by returning {@code this}.
   * @param optional An optional value for password
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableMongoShellArguments withPassword(Optional<String> optional) {
    String value = optional.orElse(null);
    if (Objects.equals(this.password, value)) return this;
    return new ImmutableMongoShellArguments(this.scriptParameters, this.scriptName, this.dbName, value, this.userName);
  }

  /**
   * Copy the current immutable object by setting a <em>present</em> value for the optional {@link MongoShellArguments#userName() userName} attribute.
   * @param value The value for userName
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableMongoShellArguments withUserName(String value) {
    String newValue = Objects.requireNonNull(value, "userName");
    if (Objects.equals(this.userName, newValue)) return this;
    return new ImmutableMongoShellArguments(this.scriptParameters, this.scriptName, this.dbName, this.password, newValue);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link MongoShellArguments#userName() userName} attribute.
   * An equality check is used on inner value to prevent copying of the same value by returning {@code this}.
   * @param optional An optional value for userName
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableMongoShellArguments withUserName(Optional<String> optional) {
    String value = optional.orElse(null);
    if (Objects.equals(this.userName, value)) return this;
    return new ImmutableMongoShellArguments(this.scriptParameters, this.scriptName, this.dbName, this.password, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableMongoShellArguments} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableMongoShellArguments
        && equalsByValue((ImmutableMongoShellArguments) another);
  }

  private boolean equalsByValue(ImmutableMongoShellArguments another) {
    return scriptParameters.equals(another.scriptParameters)
        && Objects.equals(scriptName, another.scriptName)
        && Objects.equals(dbName, another.dbName)
        && Objects.equals(password, another.password)
        && Objects.equals(userName, another.userName);
  }

  /**
   * Computes a hash code from attributes: {@code scriptParameters}, {@code scriptName}, {@code dbName}, {@code password}, {@code userName}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + scriptParameters.hashCode();
    h += (h << 5) + Objects.hashCode(scriptName);
    h += (h << 5) + Objects.hashCode(dbName);
    h += (h << 5) + Objects.hashCode(password);
    h += (h << 5) + Objects.hashCode(userName);
    return h;
  }

  /**
   * Prints the immutable value {@code MongoShellArguments} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder("MongoShellArguments{");
    builder.append("scriptParameters=").append(scriptParameters);
    if (scriptName != null) {
      builder.append(", ");
      builder.append("scriptName=").append(scriptName);
    }
    if (dbName != null) {
      builder.append(", ");
      builder.append("dbName=").append(dbName);
    }
    if (password != null) {
      builder.append(", ");
      builder.append("password=").append(password);
    }
    if (userName != null) {
      builder.append(", ");
      builder.append("userName=").append(userName);
    }
    return builder.append("}").toString();
  }

  /**
   * Creates an immutable copy of a {@link MongoShellArguments} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable MongoShellArguments instance
   */
  public static ImmutableMongoShellArguments copyOf(MongoShellArguments instance) {
    if (instance instanceof ImmutableMongoShellArguments) {
      return (ImmutableMongoShellArguments) instance;
    }
    return ImmutableMongoShellArguments.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableMongoShellArguments ImmutableMongoShellArguments}.
   * <pre>
   * ImmutableMongoShellArguments.builder()
   *    .addScriptParameters|addAllScriptParameters(String) // {@link MongoShellArguments#scriptParameters() scriptParameters} elements
   *    .scriptName(Optional&lt;String&gt;) // optional {@link MongoShellArguments#scriptName() scriptName}
   *    .dbName(Optional&lt;String&gt;) // optional {@link MongoShellArguments#dbName() dbName}
   *    .password(Optional&lt;String&gt;) // optional {@link MongoShellArguments#password() password}
   *    .userName(Optional&lt;String&gt;) // optional {@link MongoShellArguments#userName() userName}
   *    .build();
   * </pre>
   * @return A new ImmutableMongoShellArguments builder
   */
  public static ImmutableMongoShellArguments.Builder builder() {
    return new ImmutableMongoShellArguments.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableMongoShellArguments ImmutableMongoShellArguments}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private List<String> scriptParameters = new ArrayList<String>();
    private String scriptName;
    private String dbName;
    private String password;
    private String userName;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code MongoShellArguments} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(MongoShellArguments instance) {
      Objects.requireNonNull(instance, "instance");
      addAllScriptParameters(instance.scriptParameters());
      Optional<String> scriptNameOptional = instance.scriptName();
      if (scriptNameOptional.isPresent()) {
        scriptName(scriptNameOptional);
      }
      Optional<String> dbNameOptional = instance.dbName();
      if (dbNameOptional.isPresent()) {
        dbName(dbNameOptional);
      }
      Optional<String> passwordOptional = instance.password();
      if (passwordOptional.isPresent()) {
        password(passwordOptional);
      }
      Optional<String> userNameOptional = instance.userName();
      if (userNameOptional.isPresent()) {
        userName(userNameOptional);
      }
      return this;
    }

    /**
     * Adds one element to {@link MongoShellArguments#scriptParameters() scriptParameters} list.
     * @param element A scriptParameters element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addScriptParameters(String element) {
      this.scriptParameters.add(Objects.requireNonNull(element, "scriptParameters element"));
      return this;
    }

    /**
     * Adds elements to {@link MongoShellArguments#scriptParameters() scriptParameters} list.
     * @param elements An array of scriptParameters elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addScriptParameters(String... elements) {
      for (String element : elements) {
        this.scriptParameters.add(Objects.requireNonNull(element, "scriptParameters element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link MongoShellArguments#scriptParameters() scriptParameters} list.
     * @param elements An iterable of scriptParameters elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder scriptParameters(Iterable<String> elements) {
      this.scriptParameters.clear();
      return addAllScriptParameters(elements);
    }

    /**
     * Adds elements to {@link MongoShellArguments#scriptParameters() scriptParameters} list.
     * @param elements An iterable of scriptParameters elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllScriptParameters(Iterable<String> elements) {
      for (String element : elements) {
        this.scriptParameters.add(Objects.requireNonNull(element, "scriptParameters element"));
      }
      return this;
    }

    /**
     * Initializes the optional value {@link MongoShellArguments#scriptName() scriptName} to scriptName.
     * @param scriptName The value for scriptName
     * @return {@code this} builder for chained invocation
     */
    public final Builder scriptName(String scriptName) {
      this.scriptName = Objects.requireNonNull(scriptName, "scriptName");
      return this;
    }

    /**
     * Initializes the optional value {@link MongoShellArguments#scriptName() scriptName} to scriptName.
     * @param scriptName The value for scriptName
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder scriptName(Optional<String> scriptName) {
      this.scriptName = scriptName.orElse(null);
      return this;
    }

    /**
     * Initializes the optional value {@link MongoShellArguments#dbName() dbName} to dbName.
     * @param dbName The value for dbName
     * @return {@code this} builder for chained invocation
     */
    public final Builder dbName(String dbName) {
      this.dbName = Objects.requireNonNull(dbName, "dbName");
      return this;
    }

    /**
     * Initializes the optional value {@link MongoShellArguments#dbName() dbName} to dbName.
     * @param dbName The value for dbName
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder dbName(Optional<String> dbName) {
      this.dbName = dbName.orElse(null);
      return this;
    }

    /**
     * Initializes the optional value {@link MongoShellArguments#password() password} to password.
     * @param password The value for password
     * @return {@code this} builder for chained invocation
     */
    public final Builder password(String password) {
      this.password = Objects.requireNonNull(password, "password");
      return this;
    }

    /**
     * Initializes the optional value {@link MongoShellArguments#password() password} to password.
     * @param password The value for password
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder password(Optional<String> password) {
      this.password = password.orElse(null);
      return this;
    }

    /**
     * Initializes the optional value {@link MongoShellArguments#userName() userName} to userName.
     * @param userName The value for userName
     * @return {@code this} builder for chained invocation
     */
    public final Builder userName(String userName) {
      this.userName = Objects.requireNonNull(userName, "userName");
      return this;
    }

    /**
     * Initializes the optional value {@link MongoShellArguments#userName() userName} to userName.
     * @param userName The value for userName
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder userName(Optional<String> userName) {
      this.userName = userName.orElse(null);
      return this;
    }

    /**
     * Builds a new {@link ImmutableMongoShellArguments ImmutableMongoShellArguments}.
     * @return An immutable instance of MongoShellArguments
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableMongoShellArguments build() {
      return new ImmutableMongoShellArguments(createUnmodifiableList(true, scriptParameters), scriptName, dbName, password, userName);
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<? extends T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
