package de.flapdoodle.embed.mongo.transitions;

import de.flapdoodle.embed.mongo.commands.MongoImportArguments;
import de.flapdoodle.embed.mongo.commands.ServerAddress;
import de.flapdoodle.embed.process.types.ProcessArguments;
import de.flapdoodle.reverse.StateID;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link MongoImportProcessArguments}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableMongoImportProcessArguments.builder()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableMongoImportProcessArguments
    extends MongoImportProcessArguments {
  private final StateID<ProcessArguments> destination;
  private final StateID<ServerAddress> serverAddress;
  private final StateID<MongoImportArguments> arguments;

  private ImmutableMongoImportProcessArguments(ImmutableMongoImportProcessArguments.Builder builder) {
    if (builder.destination != null) {
      initShim.destination(builder.destination);
    }
    if (builder.serverAddress != null) {
      initShim.serverAddress(builder.serverAddress);
    }
    if (builder.arguments != null) {
      initShim.arguments(builder.arguments);
    }
    this.destination = initShim.destination();
    this.serverAddress = initShim.serverAddress();
    this.arguments = initShim.arguments();
    this.initShim = null;
  }

  private ImmutableMongoImportProcessArguments(
      StateID<ProcessArguments> destination,
      StateID<ServerAddress> serverAddress,
      StateID<MongoImportArguments> arguments) {
    this.destination = destination;
    this.serverAddress = serverAddress;
    this.arguments = arguments;
    this.initShim = null;
  }

  private static final byte STAGE_INITIALIZING = -1;
  private static final byte STAGE_UNINITIALIZED = 0;
  private static final byte STAGE_INITIALIZED = 1;
  private transient volatile InitShim initShim = new InitShim();

  private final class InitShim {
    private byte destinationBuildStage = STAGE_UNINITIALIZED;
    private StateID<ProcessArguments> destination;

    StateID<ProcessArguments> destination() {
      if (destinationBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (destinationBuildStage == STAGE_UNINITIALIZED) {
        destinationBuildStage = STAGE_INITIALIZING;
        StateID<ProcessArguments> computedValue = ImmutableMongoImportProcessArguments.super.destination();
        this.destination = Objects.requireNonNull(computedValue, "destination");
        destinationBuildStage = STAGE_INITIALIZED;
      }
      return this.destination;
    }

    void destination(StateID<ProcessArguments> destination) {
      this.destination = destination;
      destinationBuildStage = STAGE_INITIALIZED;
    }

    private byte serverAddressBuildStage = STAGE_UNINITIALIZED;
    private StateID<ServerAddress> serverAddress;

    StateID<ServerAddress> serverAddress() {
      if (serverAddressBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (serverAddressBuildStage == STAGE_UNINITIALIZED) {
        serverAddressBuildStage = STAGE_INITIALIZING;
        StateID<ServerAddress> computedValue = ImmutableMongoImportProcessArguments.super.serverAddress();
        this.serverAddress = Objects.requireNonNull(computedValue, "serverAddress");
        serverAddressBuildStage = STAGE_INITIALIZED;
      }
      return this.serverAddress;
    }

    void serverAddress(StateID<ServerAddress> serverAddress) {
      this.serverAddress = serverAddress;
      serverAddressBuildStage = STAGE_INITIALIZED;
    }

    private byte argumentsBuildStage = STAGE_UNINITIALIZED;
    private StateID<MongoImportArguments> arguments;

    StateID<MongoImportArguments> arguments() {
      if (argumentsBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (argumentsBuildStage == STAGE_UNINITIALIZED) {
        argumentsBuildStage = STAGE_INITIALIZING;
        StateID<MongoImportArguments> computedValue = ImmutableMongoImportProcessArguments.super.arguments();
        this.arguments = Objects.requireNonNull(computedValue, "arguments");
        argumentsBuildStage = STAGE_INITIALIZED;
      }
      return this.arguments;
    }

    void arguments(StateID<MongoImportArguments> arguments) {
      this.arguments = arguments;
      argumentsBuildStage = STAGE_INITIALIZED;
    }

    private String formatInitCycleMessage() {
      List<String> attributes = new ArrayList<>();
      if (destinationBuildStage == STAGE_INITIALIZING) attributes.add("destination");
      if (serverAddressBuildStage == STAGE_INITIALIZING) attributes.add("serverAddress");
      if (argumentsBuildStage == STAGE_INITIALIZING) attributes.add("arguments");
      return "Cannot build MongoImportProcessArguments, attribute initializers form cycle " + attributes;
    }
  }

  /**
   * @return The value of the {@code destination} attribute
   */
  @Override
  public StateID<ProcessArguments> destination() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.destination()
        : this.destination;
  }

  /**
   * @return The value of the {@code serverAddress} attribute
   */
  @Override
  public StateID<ServerAddress> serverAddress() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.serverAddress()
        : this.serverAddress;
  }

  /**
   * @return The value of the {@code arguments} attribute
   */
  @Override
  public StateID<MongoImportArguments> arguments() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.arguments()
        : this.arguments;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link MongoImportProcessArguments#destination() destination} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for destination
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableMongoImportProcessArguments withDestination(StateID<ProcessArguments> value) {
    if (this.destination == value) return this;
    StateID<ProcessArguments> newValue = Objects.requireNonNull(value, "destination");
    return new ImmutableMongoImportProcessArguments(newValue, this.serverAddress, this.arguments);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link MongoImportProcessArguments#serverAddress() serverAddress} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for serverAddress
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableMongoImportProcessArguments withServerAddress(StateID<ServerAddress> value) {
    if (this.serverAddress == value) return this;
    StateID<ServerAddress> newValue = Objects.requireNonNull(value, "serverAddress");
    return new ImmutableMongoImportProcessArguments(this.destination, newValue, this.arguments);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link MongoImportProcessArguments#arguments() arguments} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for arguments
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableMongoImportProcessArguments withArguments(StateID<MongoImportArguments> value) {
    if (this.arguments == value) return this;
    StateID<MongoImportArguments> newValue = Objects.requireNonNull(value, "arguments");
    return new ImmutableMongoImportProcessArguments(this.destination, this.serverAddress, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableMongoImportProcessArguments} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableMongoImportProcessArguments
        && equalsByValue((ImmutableMongoImportProcessArguments) another);
  }

  private boolean equalsByValue(ImmutableMongoImportProcessArguments another) {
    return destination.equals(another.destination)
        && serverAddress.equals(another.serverAddress)
        && arguments.equals(another.arguments);
  }

  /**
   * Computes a hash code from attributes: {@code destination}, {@code serverAddress}, {@code arguments}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + destination.hashCode();
    h += (h << 5) + serverAddress.hashCode();
    h += (h << 5) + arguments.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code MongoImportProcessArguments} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "MongoImportProcessArguments{"
        + "destination=" + destination
        + ", serverAddress=" + serverAddress
        + ", arguments=" + arguments
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link MongoImportProcessArguments} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable MongoImportProcessArguments instance
   */
  public static ImmutableMongoImportProcessArguments copyOf(MongoImportProcessArguments instance) {
    if (instance instanceof ImmutableMongoImportProcessArguments) {
      return (ImmutableMongoImportProcessArguments) instance;
    }
    return ImmutableMongoImportProcessArguments.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableMongoImportProcessArguments ImmutableMongoImportProcessArguments}.
   * <pre>
   * ImmutableMongoImportProcessArguments.builder()
   *    .destination(de.flapdoodle.reverse.StateID&lt;de.flapdoodle.embed.process.types.ProcessArguments&gt;) // optional {@link MongoImportProcessArguments#destination() destination}
   *    .serverAddress(de.flapdoodle.reverse.StateID&lt;de.flapdoodle.embed.mongo.commands.ServerAddress&gt;) // optional {@link MongoImportProcessArguments#serverAddress() serverAddress}
   *    .arguments(de.flapdoodle.reverse.StateID&lt;de.flapdoodle.embed.mongo.commands.MongoImportArguments&gt;) // optional {@link MongoImportProcessArguments#arguments() arguments}
   *    .build();
   * </pre>
   * @return A new ImmutableMongoImportProcessArguments builder
   */
  public static ImmutableMongoImportProcessArguments.Builder builder() {
    return new ImmutableMongoImportProcessArguments.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableMongoImportProcessArguments ImmutableMongoImportProcessArguments}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private StateID<ProcessArguments> destination;
    private StateID<ServerAddress> serverAddress;
    private StateID<MongoImportArguments> arguments;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code MongoImportProcessArguments} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(MongoImportProcessArguments instance) {
      Objects.requireNonNull(instance, "instance");
      this.destination(instance.destination());
      this.serverAddress(instance.serverAddress());
      this.arguments(instance.arguments());
      return this;
    }

    /**
     * Initializes the value for the {@link MongoImportProcessArguments#destination() destination} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link MongoImportProcessArguments#destination() destination}.</em>
     * @param destination The value for destination 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder destination(StateID<ProcessArguments> destination) {
      this.destination = Objects.requireNonNull(destination, "destination");
      return this;
    }

    /**
     * Initializes the value for the {@link MongoImportProcessArguments#serverAddress() serverAddress} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link MongoImportProcessArguments#serverAddress() serverAddress}.</em>
     * @param serverAddress The value for serverAddress 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder serverAddress(StateID<ServerAddress> serverAddress) {
      this.serverAddress = Objects.requireNonNull(serverAddress, "serverAddress");
      return this;
    }

    /**
     * Initializes the value for the {@link MongoImportProcessArguments#arguments() arguments} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link MongoImportProcessArguments#arguments() arguments}.</em>
     * @param arguments The value for arguments 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder arguments(StateID<MongoImportArguments> arguments) {
      this.arguments = Objects.requireNonNull(arguments, "arguments");
      return this;
    }

    /**
     * Builds a new {@link ImmutableMongoImportProcessArguments ImmutableMongoImportProcessArguments}.
     * @return An immutable instance of MongoImportProcessArguments
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableMongoImportProcessArguments build() {
      return new ImmutableMongoImportProcessArguments(this);
    }
  }
}
