
package de.flapdoodle.embed.mongo.client;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link AuthenticationSetup.Role}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableRole.builder()}.
 * Use the static factory method to create immutable instances:
 * {@code ImmutableRole.of()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableRole implements AuthenticationSetup.Role {
  private final String database;
  private final String collection;
  private final String name;
  private final List<String> actions;

  private ImmutableRole(String database, String collection, String name) {
    this.database = Objects.requireNonNull(database, "database");
    this.collection = Objects.requireNonNull(collection, "collection");
    this.name = Objects.requireNonNull(name, "name");
    this.actions = Collections.emptyList();
  }

  private ImmutableRole(
      String database,
      String collection,
      String name,
      List<String> actions) {
    this.database = database;
    this.collection = collection;
    this.name = name;
    this.actions = actions;
  }

  /**
   * @return The value of the {@code database} attribute
   */
  @Override
  public String database() {
    return database;
  }

  /**
   * @return The value of the {@code collection} attribute
   */
  @Override
  public String collection() {
    return collection;
  }

  /**
   * @return The value of the {@code name} attribute
   */
  @Override
  public String name() {
    return name;
  }

  /**
   * @return The value of the {@code actions} attribute
   */
  @Override
  public List<String> actions() {
    return actions;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AuthenticationSetup.Role#database() database} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for database
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableRole withDatabase(String value) {
    String newValue = Objects.requireNonNull(value, "database");
    if (this.database.equals(newValue)) return this;
    return new ImmutableRole(newValue, this.collection, this.name, this.actions);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AuthenticationSetup.Role#collection() collection} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for collection
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableRole withCollection(String value) {
    String newValue = Objects.requireNonNull(value, "collection");
    if (this.collection.equals(newValue)) return this;
    return new ImmutableRole(this.database, newValue, this.name, this.actions);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AuthenticationSetup.Role#name() name} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableRole withName(String value) {
    String newValue = Objects.requireNonNull(value, "name");
    if (this.name.equals(newValue)) return this;
    return new ImmutableRole(this.database, this.collection, newValue, this.actions);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link AuthenticationSetup.Role#actions() actions}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableRole withActions(String... elements) {
    List<String> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableRole(this.database, this.collection, this.name, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link AuthenticationSetup.Role#actions() actions}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of actions elements to set
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableRole withActions(Iterable<String> elements) {
    if (this.actions == elements) return this;
    List<String> newValue = createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableRole(this.database, this.collection, this.name, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableRole} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableRole
        && equalsByValue((ImmutableRole) another);
  }

  private boolean equalsByValue(ImmutableRole another) {
    return database.equals(another.database)
        && collection.equals(another.collection)
        && name.equals(another.name)
        && actions.equals(another.actions);
  }

  /**
   * Computes a hash code from attributes: {@code database}, {@code collection}, {@code name}, {@code actions}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + database.hashCode();
    h += (h << 5) + collection.hashCode();
    h += (h << 5) + name.hashCode();
    h += (h << 5) + actions.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Role} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Role{"
        + "database=" + database
        + ", collection=" + collection
        + ", name=" + name
        + ", actions=" + actions
        + "}";
  }

  /**
   * Construct a new immutable {@code Role} instance.
   * @param database The value for the {@code database} attribute
   * @param collection The value for the {@code collection} attribute
   * @param name The value for the {@code name} attribute
   * @return An immutable Role instance
   */
  public static ImmutableRole of(String database, String collection, String name) {
    return new ImmutableRole(database, collection, name);
  }

  /**
   * Creates an immutable copy of a {@link AuthenticationSetup.Role} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Role instance
   */
  public static ImmutableRole copyOf(AuthenticationSetup.Role instance) {
    if (instance instanceof ImmutableRole) {
      return (ImmutableRole) instance;
    }
    return ImmutableRole.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableRole ImmutableRole}.
   * <pre>
   * ImmutableRole.builder()
   *    .database(String) // required {@link AuthenticationSetup.Role#database() database}
   *    .collection(String) // required {@link AuthenticationSetup.Role#collection() collection}
   *    .name(String) // required {@link AuthenticationSetup.Role#name() name}
   *    .addActions|addAllActions(String) // {@link AuthenticationSetup.Role#actions() actions} elements
   *    .build();
   * </pre>
   * @return A new ImmutableRole builder
   */
  public static ImmutableRole.Builder builder() {
    return new ImmutableRole.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableRole ImmutableRole}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_DATABASE = 0x1L;
    private static final long INIT_BIT_COLLECTION = 0x2L;
    private static final long INIT_BIT_NAME = 0x4L;
    private long initBits = 0x7L;

    private String database;
    private String collection;
    private String name;
    private List<String> actions = new ArrayList<String>();

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Role} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(AuthenticationSetup.Role instance) {
      Objects.requireNonNull(instance, "instance");
      this.database(instance.database());
      this.collection(instance.collection());
      this.name(instance.name());
      addAllActions(instance.actions());
      return this;
    }

    /**
     * Initializes the value for the {@link AuthenticationSetup.Role#database() database} attribute.
     * @param database The value for database 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder database(String database) {
      this.database = Objects.requireNonNull(database, "database");
      initBits &= ~INIT_BIT_DATABASE;
      return this;
    }

    /**
     * Initializes the value for the {@link AuthenticationSetup.Role#collection() collection} attribute.
     * @param collection The value for collection 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder collection(String collection) {
      this.collection = Objects.requireNonNull(collection, "collection");
      initBits &= ~INIT_BIT_COLLECTION;
      return this;
    }

    /**
     * Initializes the value for the {@link AuthenticationSetup.Role#name() name} attribute.
     * @param name The value for name 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder name(String name) {
      this.name = Objects.requireNonNull(name, "name");
      initBits &= ~INIT_BIT_NAME;
      return this;
    }

    /**
     * Adds one element to {@link AuthenticationSetup.Role#actions() actions} list.
     * @param element A actions element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addActions(String element) {
      this.actions.add(Objects.requireNonNull(element, "actions element"));
      return this;
    }

    /**
     * Adds elements to {@link AuthenticationSetup.Role#actions() actions} list.
     * @param elements An array of actions elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addActions(String... elements) {
      for (String element : elements) {
        this.actions.add(Objects.requireNonNull(element, "actions element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link AuthenticationSetup.Role#actions() actions} list.
     * @param elements An iterable of actions elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder actions(Iterable<String> elements) {
      this.actions.clear();
      return addAllActions(elements);
    }

    /**
     * Adds elements to {@link AuthenticationSetup.Role#actions() actions} list.
     * @param elements An iterable of actions elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllActions(Iterable<String> elements) {
      for (String element : elements) {
        this.actions.add(Objects.requireNonNull(element, "actions element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableRole ImmutableRole}.
     * @return An immutable instance of Role
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableRole build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableRole(database, collection, name, createUnmodifiableList(true, actions));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_DATABASE) != 0) attributes.add("database");
      if ((initBits & INIT_BIT_COLLECTION) != 0) attributes.add("collection");
      if ((initBits & INIT_BIT_NAME) != 0) attributes.add("name");
      return "Cannot build Role, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<? extends T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
