package de.fayard.refreshVersions.core.internal

import de.fayard.refreshVersions.core.extensions.dom.asList
import de.fayard.refreshVersions.core.extensions.text.substringUpTo
import de.fayard.refreshVersions.core.internal.problems.Problem
import org.gradle.api.initialization.Settings
import org.w3c.dom.Document
import org.w3c.dom.Element
import org.w3c.dom.Node
import java.io.File
import javax.xml.parsers.DocumentBuilderFactory

/**
 * Android projects have Android lint enabled that will warn when the version of a dependency
 * is set to the version placeholder (`_`). This function will edit the `lint.xml` config file
 * if needed to add ignore rules for these lint warnings/errors.
 *
 * If these lint "issues" are already set to a non ignore value, they will not be edited,
 * but problems will be returned in the list instead.
 */
internal fun configureLintIfRunningOnAnAndroidProject(
    settings: Settings,
    versionsMap: Map<String, String>
): List<Problem<LintUpdatingIssue>> {
    if ("plugin.android" !in versionsMap) return emptyList()
    val lintFile = settings.rootDir.resolve("lint.xml")
    val (newXml, problems) = attemptGettingLintXmlWithMissingRules(lintFile = lintFile)
    newXml?.let {
        lintFile.writeText(it)
    }
    return problems
}

internal sealed class LintUpdatingIssue {

    class UnexpectedSeverity(
        val issueId: String,
        val actualSeverity: String?
    ) : LintUpdatingIssue() {
        override fun toString() = """
                |Expected the severity of $issueId to be "ignore" but found $actualSeverity instead.
                |If it's intentional, you can ignore this message, otherwise,
                |we recommend to correct the severity manually.
            """.trimMargin()
    }

    class ParsingFailure(
        val throwable: Throwable? = null,
        val errorMessage: String = throwable?.message
            ?: throwable?.toString()
            ?: error("Expected a throwable or an explicit message")
    ) : LintUpdatingIssue() {
        override fun toString() = errorMessage
    }

    abstract override fun toString(): String
}

@Suppress("SuspiciousCollectionReassignment")
internal fun attemptGettingLintXmlWithMissingRules(
    lintFile: File? = null,
    lintXmlContent: String = when {
        lintFile == null -> error("Need lintFile or lintXmlContent to be set")
        lintFile.exists() -> lintFile.readText()
        else -> ""
    }
): Pair<String?, List<Problem<LintUpdatingIssue>>> {

    fun nonFatalError(issue: LintUpdatingIssue): Problem<LintUpdatingIssue> {
        return Problem(level = Problem.Level.Error, issue = issue, affectedFile = lintFile)
    }

    if (lintXmlContent.isBlank()) {
        return """
            |<?xml version="1.0" encoding="UTF-8"?>
            |<lint>
            |    <!-- Generated by `./gradlew refreshVersions` to avoid errors when using _ as a version. -->
            |    <issue id="GradlePluginVersion" severity="ignore" />
            |    <issue id="GradleDependency" severity="ignore" />
            |</lint>
        """.trimMargin() to emptyList()
    }

    val document: Document = runCatching {
        DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(lintXmlContent.byteInputStream())
    }.getOrElse {
        return null to listOf(nonFatalError(issue = LintUpdatingIssue.ParsingFailure(throwable = it)))
    }

    if (document.documentElement.tagName != "lint") {
        return null to listOf(
            nonFatalError(
                issue = LintUpdatingIssue.ParsingFailure(
                    errorMessage = "Couldn't find the root tag named \"lint\""
                )
            )
        )
    }

    var problems = emptyList<Problem<LintUpdatingIssue>>()

    val nodes = document.documentElement.childNodes.asList()

    fun findSuppressNode(issueId: String): Node? = nodes.firstOrNull {
        it is Element && it.isIssue(id = issueId)
    }?.also {
        check(it is Element)
        if (it.hasAttribute("severity", expectedValue = "ignore").not()) {
            problems += Problem(
                level = Problem.Level.Warning, issue = LintUpdatingIssue.UnexpectedSeverity(
                    issueId = issueId,
                    actualSeverity = it.attributes.getNamedItem("severity")?.nodeValue
                ),
                affectedFile = lintFile
            )
        }
    }
    // Note: "AGP" stands for "Android Gradle Plugin"
    val agpErrorSuppressNodeOrNull = findSuppressNode("GradlePluginVersion")
    val libsVersionsWarningSuppressNode = findSuppressNode("GradleDependency")


    if (agpErrorSuppressNodeOrNull != null && libsVersionsWarningSuppressNode != null) return null to problems

    // We are not using the javax.xml API to edit the xml file because it messes-up the formatting by adding
    // unwanted empty lines after every tag, and working around that issue is overly complicated:
    // it requires feeding a schema, which doesn't seemed to exist, and we are not going to reverse-engineer it.
    // This text insertion logic is good enough
    val newLintXmlFile = buildString {
        val commentText = "Generated by `./gradlew refreshVersions` to avoid errors when using _ as a version."
        val firstPart = lintXmlContent.substringUpTo("<lint>")
        val secondPart = lintXmlContent.substringAfter("<lint>")
        append(firstPart)
        appendLine()
        appendLine("    <!-- $commentText -->")
        if (agpErrorSuppressNodeOrNull == null) {
            append("""    <issue id="GradlePluginVersion" severity="ignore" />""")
        }
        if (libsVersionsWarningSuppressNode == null) {
            if (agpErrorSuppressNodeOrNull == null) appendLine()
            append("""    <issue id="GradleDependency" severity="ignore" />""")
        }

        append(secondPart)
    }
    return newLintXmlFile to problems
}

private fun Element.isIssue(id: String): Boolean {
    return tagName == "issue" && hasAttribute(name = "id", expectedValue = id)
}

private fun Element.hasAttribute(name: String, expectedValue: String): Boolean {
    @Suppress("RedundantLambdaArrow") // Used to specify that the type is nullable.
    return attributes.getNamedItem(name).let { it: Node? ->
        it != null && (expectedValue == it.nodeValue)
    }
}
