package de.cidaas.oauth.interceptor;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import javax.annotation.security.DenyAll;
import javax.annotation.security.PermitAll;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.ext.Provider;

import org.apache.commons.lang3.StringUtils;
import org.jboss.resteasy.core.Headers;
import org.jboss.resteasy.core.ResourceMethodInvoker;
import org.jboss.resteasy.core.ServerResponse;
import org.jboss.resteasy.spi.ResteasyProviderFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import de.cidaas.oauth.model.TokenCheckEntity;
import de.cidaas.oauth.windowupdate.TokenWindowUpdate;

/**
 * The Class OAuthInterceptor.
 *
 * @author : thomaswidmann
 * @since: 10.12.2016
 *         <p>
 *         copyright: WidasConcepts 2016
 *         </p>
 */
@Provider
public class OAuthInterceptor implements javax.ws.rs.container.ContainerRequestFilter {

	/** The Constant LOG. */
	private static final Logger LOG = LoggerFactory.getLogger(OAuthInterceptor.class);

	/** The Constant typeJson. */
	private static final List<MediaType> typeJson = new ArrayList<MediaType>();

	/** The Constant typeXML. */
	private static final List<MediaType> typeXML = new ArrayList<MediaType>();

	static {
		typeJson.add(MediaType.APPLICATION_JSON_TYPE);
		typeXML.add(MediaType.APPLICATION_XML_TYPE);
		typeXML.add(MediaType.APPLICATION_ATOM_XML_TYPE);
		typeXML.add(MediaType.TEXT_XML_TYPE);
	}

	/** The Constant XML_RESPONSE_ERROR. */
	private static final String XML_RESPONSE_ERROR = "<error>%s</error>";

	/** The Constant JSON_RESPONSE_ERROR. */
	private static final String JSON_RESPONSE_ERROR = "{\"error\":\"%s\"}";

	/** The Constant ACCESS_DENIED. */
	private static final String ACCESS_DENIED = "Access denied for this resource";

	/** The Constant ACCESS_FORBIDDEN. */
	private static final String ACCESS_FORBIDDEN = "Nobody can access this resource";

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.ws.rs.container.ContainerRequestFilter#filter(javax.ws.rs.
	 * container. ContainerRequestContext)
	 */
	@Override
	public void filter(ContainerRequestContext requestContext) {

		LOG.info("Filtering request to: {}", requestContext.getUriInfo().getAbsolutePath().toString());

		TokenWindowUpdate tokenWindowUpdate = TokenWindowUpdate.getInstance();
		ResourceMethodInvoker methodInvoker = (ResourceMethodInvoker) requestContext
				.getProperty("org.jboss.resteasy.core.ResourceMethodInvoker");
		Method method = methodInvoker.getMethod();

		// just a hack
		if (method.getDeclaringClass().getName().equals(Constants.SWAGGER_UI_CLASS_NAME)
				&& method.getName().equals(Constants.SWAGGER_UI_METHOD_NAME)) {
			LOG.info("Swagger Request URL:" + method.getName());
			return;
		}

		if (method.isAnnotationPresent(PermitAll.class)) {
			LOG.info("Method has PermitAll:" + method.getName());
			return;
		}

		if (method.isAnnotationPresent(DenyAll.class)) {
			LOG.info("Method has DenyAll:" + method.getName());
			requestContext.abortWith(getServerResponse(methodInvoker, ACCESS_FORBIDDEN, 403));
		}

		HttpServletRequest request = ResteasyProviderFactory.getContextData(HttpServletRequest.class);
		TokenCheckEntity requestInfo = prepareHeaders(request, method);

		String accessToken = null;

		try {
			CBOAuthAccessResourceRequest oauthRequest = new CBOAuthAccessResourceRequest(request);
			accessToken = oauthRequest.getAccessToken();
			LOG.info("Access Token : {}", accessToken);
			requestInfo.setAccessToken(accessToken);
		} catch (Exception ex) {
			LOG.error("Exception {}", ex);

			requestInfo.setSuccess(false);
			tokenWindowUpdate.tokenList.add(requestInfo);

			requestContext.abortWith(getServerResponse(methodInvoker, ACCESS_DENIED, 401));
			return;
		}

		requestInfo = new TokenHelper().validateAccessToken(requestInfo, method, accessToken);

		if (requestInfo != null && requestInfo.isSuccess()) {
			OAuthUser u = new OAuthUser(accessToken, requestInfo.getUserId());
			LOG.info("Interceptor got user {} from token {}.", u.getUserId(), accessToken);
			// Push the user into the context.
			ResteasyProviderFactory.pushContext(OAuthUser.class, u);
			if (requestInfo.isNeedServerSubmit()) {
				tokenWindowUpdate.tokenList.add(requestInfo);
			}
		} else {
			LOG.error("Access denied for URL {}", requestInfo.getRequestURL());
			LOG.info("Request Info {}", requestInfo);
			if (requestInfo.isNeedServerSubmit()) {
				tokenWindowUpdate.tokenList.add(requestInfo);
			}
			requestContext.abortWith(getServerResponse(methodInvoker, ACCESS_DENIED, 401));
			return;
		}

	}

	private TokenCheckEntity prepareHeaders(HttpServletRequest request, Method method) {

		TokenCheckEntity tE = new TokenCheckEntity();

		String referrer = request.getHeader("referrer");
		if (StringUtils.isNotEmpty(referrer)) {
			tE.setReferrer(referrer);
		}
		String ipAddress = request.getHeader("X-FORWARDED-FOR");
		if (ipAddress == null) {
			ipAddress = request.getRemoteAddr();
		}
		if (ipAddress != null && ipAddress.contains(",")) {
			ipAddress = ipAddress.split(",")[0];
		}
		if (StringUtils.isNotEmpty(ipAddress)) {
			tE.setIpAddress(ipAddress);

		}

		String host = request.getHeader("X-Forwarded-Host");
		if (host == null) {
			host = request.getRemoteHost();
		}
		if (StringUtils.isNotEmpty(host)) {
			tE.setHost(host);
		}

		String langString = request.getHeader("Accept-Language");
		if (StringUtils.isNotEmpty(langString)) {
			tE.setAcceptLanguage(langString);
		}

		String browserInfo = request.getHeader("user-agent");
		if (StringUtils.isNotEmpty(browserInfo)) {
			tE.setUserAgent(browserInfo);
		}

		String requestURL = request.getRequestURL().toString();
		if (StringUtils.isNotEmpty(requestURL)) {
			tE.setRequestURL(requestURL);
		}

		TokenHelper th = new TokenHelper();
		tE.setRequestedScopes(th.getAnotationRequestedScopeJoined(method));
		tE.setRequestedRoles(th.getAnotationRequestedRolesJoined(method));

		tE.setRequestInfo(new HashMap<>());

		List<String> allHeaders = Collections.list(request.getHeaderNames());

		for (String headerKey : allHeaders) {
			tE.getRequestInfo().put(headerKey, request.getHeader(headerKey));
		}

		return tE;
	}

	/**
	 * Generate a dynamic {@link ServerResponse} to output an error message
	 * according to the invokers @produces annotation.
	 *
	 * @param methodInvoker
	 *            the invoker
	 * @param entity
	 *            the message
	 * @param status
	 *            the html status code
	 * @return the {@link ServerResponse}
	 */
	private ServerResponse getServerResponse(ResourceMethodInvoker methodInvoker, Object entity, int status) {
		if (methodInvoker.doesProduce(typeJson)) {
			return new ServerResponse(String.format(JSON_RESPONSE_ERROR, entity), status, new Headers<>());
		} else if (methodInvoker.doesProduce(typeXML)) {
			return new ServerResponse(String.format(XML_RESPONSE_ERROR, entity), status, new Headers<>());
		}
		return new ServerResponse(entity, status, new Headers<>());
	}

}
