package de.cidaas.oauth.util;


import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;


import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

/**
 * Configuration loader to load java properties either as single property or as whole {@link Properties} object
 * 
 * Note: warc.homepath must be set as System.property
 * 
 * @author jan-henrik.preuss
 *
 */
public class ConfigurationLoader {

  private static final Logger logger = LoggerFactory.getLogger(ConfigurationLoader.class);
  private static final String path = System.getProperty("warc.homepath");

  /**
   * Loads the given properties file
   * 
   * @param pFilename
   *          Property File
   * @return null if file dose not exist
   */
  public static Properties getPropertiesOfFile(String pFilename) {
	if( StringUtils.isEmpty( path ) ){
		throw new RuntimeException( "warc.homepath not set while reading Propertyfile:" + pFilename );
	}
	if( StringUtils.isEmpty( pFilename ) ){
		throw new RuntimeException( "No properties filename to read from specified, warc.homepath=" + path );
	}
	
    try {
      Properties props = new Properties();
      InputStream in;
      in = getFullqualifiedFilename(pFilename);
      props.load(in);
      
      for (Object key: props.keySet()) {
    	  String propValue = System.getenv(key.toString());
    	  if (propValue != null){
    		  props.setProperty(key.toString(), propValue);
    	  }
      }
      
      in.close();
      return props;
    } catch (FileNotFoundException e) {
      logger.error(pFilename + " not found.", e);
      return null;
    } catch (IOException e) {
      logger.error("IOException while loading " + pFilename, e);
      return null;
    }
  }


  private static InputStream getFullqualifiedFilename(String pFilename) throws FileNotFoundException {
    StringBuilder lSB = new StringBuilder(path);
    if (path != null && !path.endsWith("/"))
      lSB.append("/");
    lSB.append("conf/");
    lSB.append(pFilename);
    final String lFilePath = lSB.toString();
    logger.debug("load properties from file:" + lFilePath);
    InputStream inputStream = ConfigurationLoader.class.getClassLoader().getResourceAsStream(lFilePath);
    if (inputStream == null) {
      inputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream(lFilePath);
    }
    if (inputStream == null) {
      inputStream = new FileInputStream(lFilePath);
    }
    return inputStream;
  }

  /**
   * Get a single property from baufi.properties
   * 
   * @param key
   *          the property key
   * @param defaultValue
   *          the default value
   * @param pFilename
   *          Property File in your warc.homepath/conf folder
   * @return the defaultValue if property dose not exist
   */
  public static String getPropertyByKey(String key, String defaultValue, String pFilename) {
    Properties properties = getPropertiesOfFile(pFilename);
    if (properties != null) {
      return properties.getProperty(key);
    }
    return defaultValue;
  }

  /**
   * Get a single property from baufi.properties
   * 
   * @param key
   *          the property key
   * @param pFilename
   *          Property File in your warc.homepath/conf folder
   * @return null if property dose not exist
   */
  public static String getPropertyByKey(String key, String pFilename) {
    return getPropertyByKey(key, null, pFilename);
  }
  public static Properties getPropertiesFile() {
		Properties prop = new Properties();
		InputStream input = null;

		try {
			
			input = ConfigurationLoader.class.getClassLoader()
					.getResourceAsStream("cidaas_config.properties");

			// load a properties file
			prop.load(input);

		} catch (IOException ex) {
			ex.printStackTrace();
		} finally {
			if (input != null) {
				try {
					input.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
		return prop;
	}

}
