package de.cidaas.oauth.model;


import javax.servlet.http.HttpServletRequest;

import org.apache.oltu.oauth2.common.error.OAuthError;
import org.apache.oltu.oauth2.common.exception.OAuthProblemException;
import org.apache.oltu.oauth2.common.exception.OAuthSystemException;
import org.apache.oltu.oauth2.common.message.types.ParameterStyle;
import org.apache.oltu.oauth2.common.utils.OAuthUtils;
import org.apache.oltu.oauth2.common.validators.OAuthValidator;
import org.apache.oltu.oauth2.rs.ResourceServer;
import org.apache.oltu.oauth2.rs.extractor.TokenExtractor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import de.cidaas.oauth.interceptor.BearerResourceServer;



/**
 * The Class OAuthAccessResourceRequest.
 *
 * @author Michael Scheuner &lt;michael.scheuner@widas.de&gt;
 */
public class OAuthAccessResourceRequest {

	/** The Constant LOG. */
	private static final Logger LOG = LoggerFactory.getLogger(OAuthAccessResourceRequest.class);

	/** The request. */
	private final HttpServletRequest request;

	/** The used resource server. */
	private ResourceServer usedResourceServer;

	/** The extractor. */
	private TokenExtractor extractor;

	/** The parameter styles. */
	private final ParameterStyle[] parameterStyles = { ParameterStyle.HEADER, ParameterStyle.BODY, ParameterStyle.QUERY };

	/**
	 * Instantiates a new CBO auth access resource request.
	 *
	 * @param request
	 *          the request
	 * @throws OAuthSystemException
	 *           the o auth system exception
	 * @throws OAuthProblemException
	 *           the o auth problem exception
	 */
	public OAuthAccessResourceRequest(HttpServletRequest request) throws OAuthSystemException, OAuthProblemException {
		this.request = request;
		validate();
	}

	/**
	 * Gets the access token.
	 *
	 * @return the access token“
	 * @throws OAuthSystemException
	 *           the o auth system exception
	 */
	public String getAccessToken() throws OAuthSystemException {
		return extractor.getAccessToken(request);
	}

	
	/**
	 * Validate.
	 *
	 * @throws OAuthSystemException
	 *           the o auth system exception
	 * @throws OAuthProblemException
	 *           the o auth problem exception
	 */
	@SuppressWarnings("unchecked")
	private void validate() throws OAuthSystemException, OAuthProblemException {

		int foundValidStyles = 0;
		boolean lackAuthInfo = false;
		String lackAuthReason = "OAuth parameters were not found";
		ResourceServer resourceServer = new BearerResourceServer();
		ParameterStyle usedStyle = null;
		for (ParameterStyle style : parameterStyles) {
			// We are ok with at least one style. Header preferred.
			if (foundValidStyles > 0) {
				continue;
			}
			try {
				@SuppressWarnings("rawtypes")
				OAuthValidator validator = resourceServer.instantiateValidator(style);
				validator.validateContentType(request);
				validator.validateMethod(request);
				validator.validateRequiredParameters(request);
				usedResourceServer = resourceServer;
				usedStyle = style;
				foundValidStyles++;
			} catch (OAuthProblemException e) {
				LOG.error("Validation did nod succeed whith style: {}, {}", style, e.getMessage());
				// request lacks any authentication information?
				if (OAuthUtils.isEmpty(e.getError())) {
					lackAuthInfo = true;
					lackAuthReason = e.getDescription();
				}
			}
		}
		if (foundValidStyles > 1) {
			throw OAuthProblemException.error(OAuthError.TokenResponse.INVALID_REQUEST,
					"Found more than one mechanism for authenticating client");
		}

		if (foundValidStyles == 0 && lackAuthInfo) {
			throw OAuthProblemException.error(null, lackAuthReason);
		}

		if (foundValidStyles == 0) {
			throw OAuthProblemException.error(OAuthError.TokenResponse.INVALID_REQUEST, "OAuth parameters were not found");
		}

		extractor = usedResourceServer.instantiateExtractor(usedStyle);
	}
}

