(ns reacl-c.async
  (:require [reacl-c.core :as c #?@(:cljs [:include-macros true])]
            [clojure.core.async :as c-async]
            [active.clojure.functions :as f]
            #?(:cljs [active.clojure.cljs.record :as r :include-macros true])
            #?(:clj [active.clojure.record :as r])))

(c/defn-effect ^:private reset-atom! [atom v]
  (reset! atom v))

(c/defn-effect ^:private push! [channel v]
  (c-async/put! channel v #_(fn success [])))

(defn- send-act [[state l-state] act]
  ;; aka 'receive action' from below.
  (c/return :action (push! (:actions l-state) act)))

(defn- handle-actions [[state l-state] act]
  (cond
    (and (vector? act)
         (= ::update-state (first act)))
    (let [[f args] (second act)]
      (c/return :state [(apply f state args) l-state]))
    
    (and (vector? act)
         (= ::show (first act)))
    (c/return :state [state (assoc l-state :item (second act))])
    
    :else
    (send-act [state l-state] act)))

(defn- send-msg [[state l-state] msg]
  (c/return :action (push! (:messages l-state) msg)))

(defn- runtime [return! [state l-state]]
  (c/handle-action (c/focus c/second-lens
                            (:item l-state))
                   handle-actions))

(r/define-record-type ^:private AsyncItem
  (make-this return! state item messages actions)
  this?
  [return! this-return!
   state this-state
   item this-item
   messages this-messages
   actions this-actions])

(defn get-state
  "Returns the current state of this async item."
  [this]
  @(this-state this))

(defn update-state!
  "Updates the current state of this async item, to the result of `(f
  state & args)`."
  [this f & args]
  ((this-return! this) (c/return :action [::update-state [f args]])))

(defn set-state!
  "Sets the current state of this async item. If the new state depends
  on the previous state, then use [[update-state!]] instead."
  [this state]
  (update-state! this (f/constantly state)))

(defn show!
  "Replaces the item that this async item currently renders as."
  [this item]
  ;; TODO: return a channel that closes after the item become visible (on-mount close!...)
  ((this-return! this) (c/return :action [::show item])))

(defn send-message!
  "Sends a message to the item specified by the given reference."
  [this ref msg]
  ((this-return! this) (c/return :message [ref msg])))

(defn emit-action!
  "Emits an action from this async item."
  [this action]
  ((this-return! this) (c/return :action action)))

(defn messages
  "Returns the core.async channel containing all messages sent to
  this async item."
  [this]
  (this-messages this))

(defn actions
  "Returns the core.async channel containing all actions emitted by
  the item this async item renders to (see [[show!]])."
  [this]
  (this-actions this))

;; TODO: dynamic?

;; TODO: with-ref...? install-message-pipeline! ...

(c/defn-effect ^:private init! [f this args]
  (apply f this args))

(defn- init [f args return! [state l-state]]
  ;; TODO: restart if f or args change?!
  (if (nil? l-state)
    (let [state-atom (atom state) ;; TODO: create these in an effect!?
          messages (c-async/chan) ;; TODO: make do something for 'unresponsive' items - a warning at least.
          actions (c-async/chan)
          checked-return!
          (fn [ret]
            ;; make return! return false if unmounted
            (if (:mounted? @state-atom)
              (do (return! ret)
                  true)
              false))
          this (make-this checked-return! state-atom c/empty messages actions)]
      (c/return :action (init! f this args)
                :state [state {:item c/empty
                               :state state-atom
                               :messages messages
                               :actions actions}]))
    (c/return)))

(c/defn-effect ^:private finish! [l-state]
  (c-async/close! (:messages l-state))
  (c-async/close! (:actions l-state)))

(defn- finish [[state l-state]]
  (c/return :action (finish! l-state)))

(defn- setup [f args return!]
  (c/handle-message (f/partial send-msg)
                    (c/fragment (c/once (f/partial init f args return!)
                                        finish)
                                (c/dynamic (f/partial runtime)))))

(def ^:private copy-to-local-atom 
  (c/once (fn [[state l-state]]
            (c/return :action (reset-atom! (:state l-state) state)))))

(defn async
  "Returns an async item, where `(f this & args)` is evaluated once, which should start a core.async thread."
  [f & args]
  ;; TODO: make receiving/capturing messages optional/an opt-in?
  ;; TODO: make capturing actions optional/an opt-in?
  ;; TODO: allow capturing new state/state changes? (No!?)
  ;; TODO: when the thread ends, make it c/empty again (also throw if it causes an error?)
  (c/local-state nil
                 (c/fragment
                  copy-to-local-atom
                  (c/with-async-return (f/partial setup f args)))))


(comment
  
  (async (fn [this]
           (c-async/go
             (show! this (c/fragment #_item-list
                                     #_new-item-control))
             #_(let [resp (c-async/<! (ajax ...))]
                 (show! (response resp))
                 )
             (loop []
               (case (c-async/<! (actions this))
                 ::new-item nil #_(update-state! this conj ...
                                                 ...
                                                 )
                 )
               (recur))))))

(comment
  ;; 1. init item
  ;; 1b. init state? (-> once? deinit?)
  ;; 2. await action from item
  ;; 2b. await message
  ;; 2c. await state change from item
  ;; 3. emit action
  ;; 3b. send message
  ;; 4. set item, set state.

  )
