package dc.android.common.handler;

import java.util.Date;

import android.content.Context;
import android.os.Looper;
import dc.android.bridge.R;
import dc.android.common.BridgeContext;
import dc.android.common.utils.SharePreferencesUtils;
import dc.android.common.utils.TaskUtils;
import dc.android.common.utils.ThrowableUtils;
import dc.common.Logger;
import dc.common.utils.StringUtils;

import static dc.android.common.BridgeContext.CRASH_INFO_MAX;
import static dc.android.common.BridgeContext.LINE;

/**
 * splash启用后，会引发重启功能
 * 故不启用此类，亦可重启
 *
 * @author senrsl
 * @ClassName: CrashHandler
 * @Package: dc.android.common.handler
 * @CreateTime: 2018/7/26 下午7:59
 */
public class CrashHandler implements Thread.UncaughtExceptionHandler {
    // 系统默认的UncaughtException处理类
    private Thread.UncaughtExceptionHandler mDefaultHandler;
    // CrashHandler实例
    private static CrashHandler INSTANCE = new CrashHandler();
    // 程序的Context对象
    private Context ctx;
    // 用来存储设备信息和异常信息
//    private Map<String, String> infos = new HashMap<String, String>();
//    // 用于格式化日期,作为日志文件名的一部分
//    private DateFormat formatter = new SimpleDateFormat("yyyy-MM-dd-HH-mm-ss");

    private Callback cb;

    private long initTime;//启动时间

    /**
     * 保证只有一个CrashHandler实例
     */
    private CrashHandler() {
    }

    /**
     * 获取CrashHandler实例 ,单例模式
     */
    public static CrashHandler getInstance() {
        return INSTANCE;
    }


    public void init(Context context) {
        init(context, null);
    }

    /**
     * 初始化
     *
     * @param context
     */
    public void init(Context context, Callback cb) {
        ctx = context;
        this.cb = cb;
        this.initTime = getCurrentTime();
        // 获取系统默认的UncaughtException处理器
        mDefaultHandler = Thread.getDefaultUncaughtExceptionHandler();
        // 设置该CrashHandler为程序的默认处理器
        Thread.setDefaultUncaughtExceptionHandler(this);
    }

    /**
     * 当UncaughtException发生时会转入该函数来处理
     */
    @Override
    public void uncaughtException(Thread thread, Throwable ex) {
        if (!handleException(ex) && mDefaultHandler != null) {
            // 如果用户没有处理则让系统默认的异常处理器来处理
            mDefaultHandler.uncaughtException(thread, ex);
        } else {
            TaskUtils.restart(ctx);
        }
    }

//    public Class<?> getTopActivity() {
//        ActivityManager manager = (ActivityManager) mContext.getSystemService(Context.ACTIVITY_SERVICE);
//        String className = manager.getRunningTasks(1).get(0).topActivity.getClassName();
//        Class<?> cls = null;
//        try {
//            cls = Class.forName(className);
//        } catch (ClassNotFoundException e) {
//            e.printStackTrace();
//        }
//        return cls;
//    }

    /**
     * 自定义错误处理,收集错误信息 发送错误报告等操作均在此完成.
     *
     * @return true:如果处理了该异常信息;否则返回false.
     */
    private boolean handleException(Throwable ex) {
        if (ex == null) {
            return false;
        }
        // 使用Toast来显示异常信息
        showRestartTips();
        // 收集设备参数信息
//        collectDeviceInfo(mContext);
        // 保存日志文件
        String info = ThrowableUtils.getInfo(ex);
        saveCrashInfo2File(info);
        log2Fabric(ex, info);
        return true;
    }

    private void showRestartTips() {
        String tips = ctx.getString(R.string.restart);
        if (StringUtils.isEmpty(tips)) return;

        new Thread() {
            @Override
            public void run() {
                Looper.prepare();
                Logger.w(ctx, ctx.getString(R.string.restart));
                Looper.loop();
            }
        }.start();
    }

    private void saveCrashInfo2File(String s) {
        if (s.length() > CRASH_INFO_MAX) s = s.substring(0, CRASH_INFO_MAX);
        StringBuffer sb = new StringBuffer();
        sb.append(getTime(initTime)).append(BridgeContext.TAB);
        sb.append(getTime(getCurrentTime())).append(BridgeContext.TAB);
        sb.append(getCurrentTime() - initTime).append(LINE);
        sb.append(s);
        new SharePreferencesUtils(ctx).saveSharedPreferencesValue(BridgeContext.KEY_CRASH, sb.toString());
    }

    private void log2Fabric(Throwable ex, String info) {
        //CrashUtils.log(ex);
        if (null != cb) cb.onFabric(ex, info);
    }

    private String getTime(long timestamp) {
        return new Date(timestamp).toString();
    }

    private long getCurrentTime() {
        return System.currentTimeMillis();
    }


//    /**
//     * 保存错误信息到文件中，需要有对SD的读写权限！
//     *
//     * @param ex
//     * @return 返回文件名称, 便于将文件传送到服务器
//     */
//    private String saveCrashInfo2File(Throwable ex) {
//        StringBuffer sb = new StringBuffer();
//        for (Map.Entry<String, String> entry : infos.entrySet()) {
//            String key = entry.getKey();
//            String value = entry.getValue();
//            sb.append(key + "=" + value + "\n");
//        }
//        Writer writer = new StringWriter();
//        PrintWriter printWriter = new PrintWriter(writer);
//        ex.printStackTrace(printWriter);
//        Throwable cause = ex.getCause();
//        while (cause != null) {
//            cause.printStackTrace(printWriter);
//            cause = cause.getCause();
//        }
//        printWriter.close();
//        String result = writer.toString();
//        sb.append(result);
//        try {
//            long timestamp = System.currentTimeMillis();
//            String time = formatter.format(new Date());
//            String fileName = "crash-" + time + "-" + timestamp + ".log"; // 崩溃日志的文件
//            if (Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED)) {
//                String path = "/sdcard/crash/"; // 崩溃日志的存储路径
//                File dir = new File(path);
//                if (!dir.exists()) {
//                    dir.mkdirs();
//                }
//                FileOutputStream fos = new FileOutputStream(path + fileName);
//                fos.write(sb.toString().getBytes());
//                fos.close();
//            }
//            return fileName;
//        } catch (Exception e) {
//            Log.e(TAG, "an error occured while writing file...", e);
//        }
//        return null;
//
//    }

    public interface Callback {
        /**
         * 不要在此实现中做耗时(如请求网络)操作
         *
         * @param ex
         * @param info
         */
        void onFabric(Throwable ex, String info);
    }

}
