package com.zoyi.channel.plugin.android.view.youtube.player.views;


import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Bitmap;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.View;
import android.webkit.WebChromeClient;
import android.webkit.WebSettings;
import android.webkit.WebView;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.util.Initializer;
import com.zoyi.channel.plugin.android.view.youtube.player.YouTubePlayer;
import com.zoyi.channel.plugin.android.view.youtube.player.YouTubePlayerBridge;
import com.zoyi.channel.plugin.android.view.youtube.player.listener.YouTubePlayerListener;
import com.zoyi.channel.plugin.android.view.youtube.player.option.IFramePlayerOptions;
import com.zoyi.channel.plugin.android.view.youtube.player.util.Utils;
import com.zoyi.rx.functions.Action1;

import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;

/**
 * WebView implementation of [YouTubePlayer]. The player runs inside the WebView, using the IFrame Player API.
 */
public class WebViewYouTubePlayer extends WebView implements YouTubePlayer, YouTubePlayerBridge.YouTubePlayerBridgeCallbacks {

  @Nullable
  private Action1<YouTubePlayer> youTubePlayerInitListener;

  private Set<YouTubePlayerListener> youTubePlayerListeners;
  private Handler mainThreadHandler;

  private boolean isBackgroundPlaybackEnabled = false;

  public WebViewYouTubePlayer(Context context) {
    super(context);
    init();
  }

  public WebViewYouTubePlayer(Context context, AttributeSet attrs) {
    super(context, attrs);
    init();
  }

  public WebViewYouTubePlayer(Context context, AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    init();
  }

  public boolean isBackgroundPlaybackEnabled() {
    return isBackgroundPlaybackEnabled;
  }

  public void setBackgroundPlaybackEnabled(boolean backgroundPlaybackEnabled) {
    isBackgroundPlaybackEnabled = backgroundPlaybackEnabled;
  }

  @Initializer
  private void init() {
    youTubePlayerListeners = new HashSet<>();

    mainThreadHandler = new Handler(Looper.getMainLooper());
  }

  public void initialize(Action1<YouTubePlayer> initListener, @Nullable IFramePlayerOptions playerOptions) throws IOException {
    youTubePlayerInitListener = initListener;

    initWebView(playerOptions == null ? IFramePlayerOptions.DEFAULT : playerOptions);
  }

  @Override
  public void onYouTubeIFrameAPIReady() {
    if (youTubePlayerInitListener != null) {
      youTubePlayerInitListener.call(this);
    }
  }

  @Override
  public YouTubePlayer getInstance() {
    return this;
  }

  @Override
  public void loadVideo(@Nullable String videoId, float startSeconds) {
    if (videoId != null) {
      mainThreadHandler.post(() -> loadUrl(String.format(Locale.ENGLISH, "javascript:loadVideo('%s', %f)", videoId, startSeconds)));
    }
  }

  @Override
  public void cueVideo(@Nullable String videoId, float startSeconds) {
    if (videoId != null) {
      mainThreadHandler.post(() -> loadUrl(String.format(Locale.ENGLISH, "javascript:loadVideo('%s', %f)", videoId, startSeconds)));
    }
  }

  @Override
  public void play() {
    mainThreadHandler.post(() -> loadUrl("javascript:playVideo()"));
  }

  @Override
  public void pause() {
    mainThreadHandler.post(() -> loadUrl("javascript:pauseVideo()"));
  }

  @Override
  public void mute() {
    mainThreadHandler.post(() -> loadUrl("javascript:mute()"));
  }

  @Override
  public void unMute() {
    mainThreadHandler.post(() -> loadUrl("javascript:unMute()"));
  }

  @Override
  public void setVolume(int volumePercent) {
    if (volumePercent < 0 || volumePercent > 100) {
      throw new IllegalArgumentException("Volume must be between 0 and 100");
    }

    mainThreadHandler.post(() -> loadUrl(String.format(Locale.ENGLISH, "javascript:setVolume(%d)", volumePercent)));
  }

  @Override
  public void seekTo(float time) {
    mainThreadHandler.post(() -> loadUrl(String.format(Locale.ENGLISH, "javascript:seekTo(%f)", time)));
  }

  @Override
  public void destroy() {
    youTubePlayerListeners.clear();
    mainThreadHandler.removeCallbacksAndMessages(null);
    super.destroy();
  }

  @Override
  public Collection<YouTubePlayerListener> getListeners() {
    return new HashSet<>(youTubePlayerListeners);
  }

  @Override
  public boolean addListener(YouTubePlayerListener listener) {
    return youTubePlayerListeners.add(listener);
  }

  @Override
  public boolean removeListener(YouTubePlayerListener listener) {
    return youTubePlayerListeners.remove(listener);
  }

  @SuppressLint({ "SetJavaScriptEnabled", "AddJavascriptInterface" })
  private void initWebView(IFramePlayerOptions playerOptions) throws IOException {
    getSettings().setJavaScriptEnabled(true);
    getSettings().setCacheMode(WebSettings.LOAD_NO_CACHE);

    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
      getSettings().setMediaPlaybackRequiresUserGesture(false);
    }

    addJavascriptInterface(new YouTubePlayerBridge(this), "YouTubePlayerBridge");

    String htmlPage = Utils
        .readHTMLFromUTF8File(getResources().openRawResource(R.raw.ch_plugin_ayp_youtube_player))
        .replace("<<injectedPlayerVars>>", playerOptions.toString());

    loadDataWithBaseURL(playerOptions.getOrigin(), htmlPage, "text/html", "utf-8", null);

    // if the video's thumbnail is not in memory, show a black screen
    setWebChromeClient(new WebChromeClient() {
      @Override
      public Bitmap getDefaultVideoPoster() {
        Bitmap result = super.getDefaultVideoPoster();

        return result == null ? Bitmap.createBitmap(1, 1, Bitmap.Config.RGB_565) : result;
      }
    });
  }

  @Override
  public void onWindowVisibilityChanged(int visibility) {
    if (isBackgroundPlaybackEnabled && (visibility == View.GONE || visibility == View.INVISIBLE)) {
      return;
    }

    super.onWindowVisibilityChanged(visibility);
  }
}