package com.zoyi.channel.plugin.android.activity.lounge;

import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.action.*;
import com.zoyi.channel.plugin.android.activity.common.userchat.model.ChatItem;
import com.zoyi.channel.plugin.android.activity.common.userchat.model.UserChatItem;
import com.zoyi.channel.plugin.android.activity.common.watermark.OnWatermarkCounterCompleteListener;
import com.zoyi.channel.plugin.android.activity.common.watermark.WatermarkCounter;
import com.zoyi.channel.plugin.android.activity.lounge.contract.LoungeAdapterContract;
import com.zoyi.channel.plugin.android.activity.lounge.contract.LoungeContract;
import com.zoyi.channel.plugin.android.activity.lounge.enumerate.PreviewState;
import com.zoyi.channel.plugin.android.base.AbstractAdapterPresenter;
import com.zoyi.channel.plugin.android.bind.BindAction;
import com.zoyi.channel.plugin.android.enumerate.FetchState;
import com.zoyi.channel.plugin.android.enumerate.SocketStatus;
import com.zoyi.channel.plugin.android.global.Api;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.Plugin;
import com.zoyi.channel.plugin.android.selector.*;
import com.zoyi.channel.plugin.android.store.*;
import com.zoyi.channel.plugin.android.util.ResUtils;

import java.util.Collections;

public class LoungePresenter
    extends AbstractAdapterPresenter<LoungeContract.View, LoungeAdapterContract.View, LoungeAdapterContract.Model>
    implements LoungeContract.Presenter, OnWatermarkCounterCompleteListener {

  @Nullable
  private String topActivityName;

  private WatermarkCounter watermarkCounter;

  public LoungePresenter(
      LoungeContract.View view,
      LoungeAdapterContract.View adapterView,
      LoungeAdapterContract.Model adapterModel,
      @Nullable String topActivityName
  ) {
    super(view, adapterView, adapterModel);
    this.topActivityName = topActivityName;
    this.watermarkCounter = new WatermarkCounter(this);

    ChannelSelector.bindChannel(channel -> {
      if (!isRunning(BindAction.FETCH_LOUNGE) && watermarkCounter.isComplete()) {
        view.handleWatermarkVisibility();
      }
    }).bind(this);
  }

  // Fetch methods

  @Override
  public void fetchBundleData() {
    fetchPreviewData();
  }

  @Override
  public void fetchPreviewData() {
    fetchLoungeData();
  }

  private void fetchUserChats(boolean showClosedChats) {
    UserChatAction.fetchUserChats(showClosedChats);
  }

  private void fetchLoungeData() {
    Plugin plugin = PluginStore.get().pluginState.get();

    resetLounge();

    if (plugin != null) {

      LoungeAction.fetchLounge(plugin.getId(), topActivityName, (loungeMediaTypes) -> {
        if (loungeMediaTypes != null && !loungeMediaTypes.isEmpty()) {
          this.watermarkCounter.active(loungeMediaTypes.size());

          for (String type : loungeMediaTypes) {
            fetchLoungeMediaData(type);
          }
        } else {
          this.watermarkCounter.active(0);
        }
      }).bind(this, BindAction.FETCH_LOUNGE);
    }
  }

  @Override
  public void fetchLoungeMediaData(String type) {
    Plugin plugin = PluginStore.get().pluginState.get();

    if (plugin != null && Const.APP_MEDIA_INSTAGRAM.equals(type)) {
      // need to add lounge media api when new lounge media added
      view.onLoungeMediaStateChange(type, FetchState.LOADING);

      // do not move watermarkCounter.next(type) in onComplete because of timing issue
      Api.getLoungeMedia(plugin.getId(), Collections.singletonList(type))
          .onComplete(() -> this.watermarkCounter.next(type))
          .onError(ex -> view.onLoungeMediaStateChange(type, FetchState.FAILED))
          .call(repo -> {
            if (!repo.getLoungeMedia().isEmpty()) {
              view.onLoungeMediaStateChange(type, FetchState.COMPLETE);
              view.onFetchLoungeMediaInstagram(repo.getLoungeMedia());
            } else {
              view.onLoungeMediaStateChange(type, FetchState.EMPTY);
            }
          })
          .bind(this, BindAction.GET_LOUNGE_MEDIA_INSTAGRAM);
    } else {
      this.watermarkCounter.next(type);
    }
  }

  // bind

  @Override
  public void init() {
    if (SocketStore.get().socketState.get() != SocketStatus.READY) {
      view.onPreviewStateChange(PreviewState.FAILED);
    }

    bind(LoungeSelector.bindPreviewState(previewState -> view.onPreviewStateChange(previewState)));

    bind(ProfileSelector.bindDefaultProfile(profile -> view.onDefaultProfileChange(profile)));

    bind(LoungeSelector.bindWelcomeMessage(plaintText -> view.onWelcomeMessageChange(plaintText)));

    bind(SettingsSelector.bindShowClosedChats(showClosedChats -> {
      if (showClosedChats != null) {
        fetchUserChats(showClosedChats);
      }
    }));

    bind(ChatsSelector.bindChats((activeChatItems, closedChatItems) -> adapterModel.updateChatItems(activeChatItems)));

    bind(SocketSelector.bindSocket(socketStatus -> {
      if (socketStatus == SocketStatus.READY) {
        fetchBundleData();
      }
    }));
  }

  @Override
  public void removeChat(ChatItem item) {
    if (item instanceof UserChatItem) {
      view.showProgress(ResUtils.getString("ch.chat.delete_progress"));

      ChatAction.removeUserChat(
          item.getId(),
          (userChatRepo) -> {
            view.hideProgress();
            userChatRepo.remove();
            UserChatStore.get().userChats.removeByKey(item.getId());
          },
          () -> view.hideProgress()
      );
    }
  }

  @Override
  public void fetchConnect(String appMessengerName) {
    AppMessengerAction.fetchConnect(appMessengerName);
  }

  @Override
  public void onComplete() {
    view.handleWatermarkVisibility();
  }

  private void resetLounge() {
    // need to unbind all lounge media apis
    unbind(BindAction.GET_LOUNGE_MEDIA_INSTAGRAM);

    this.watermarkCounter.reset();

    view.deactivateWatermark();
  }
}
