package com.zoyi.channel.plugin.android.activity.chat.viewholder;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.view.View;
import android.widget.LinearLayout;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.chat.listener.*;
import com.zoyi.channel.plugin.android.activity.chat.view.attachment.AttachmentsLayout;
import com.zoyi.channel.plugin.android.activity.chat.view.viewholder.*;
import com.zoyi.channel.plugin.android.bind.Binder;
import com.zoyi.channel.plugin.android.enumerate.TranslationState;
import com.zoyi.channel.plugin.android.model.etc.TranslationInfo;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.selector.TranslationSelector;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.com.annimon.stream.Stream;

import java.util.ArrayList;
import java.util.List;

public abstract class BaseMessageHolder
    extends AbsMessageHolder
    implements
    OnBaseMessageActionListener,
    OnWebPageContentActionListener,
    OnAttachmentContentActionListener,
    OnLinkButtonActionListener,
    View.OnLongClickListener {

  protected AbsMessageTextView textMessage;

  protected Context context;

  private LinearLayout rootLayout;
  private AttachmentsLayout viewMessageContentAttachments;
  private WebPageContentView viewMessageContentWebPage;
  private TranslationContentView viewMessageContentTranslation;
  private ReactionsView viewMessageContentReactions;
  private GiantEmojiTextView viewMessageContentGiantEmoji;

  private List<MessageContentView> contents;

  @Nullable
  private OnBaseMessageActionListener listener;

  @Nullable
  private Binder translationBinder;

  BaseMessageHolder(View itemView, @Nullable OnBaseMessageActionListener listener) {
    super(itemView);

    this.context = itemView.getContext();
    this.listener = listener;

    rootLayout = itemView.findViewById(R.id.ch_rootMessageHolder);
    rootLayout.setOnLongClickListener(this);

    textMessage = itemView.findViewById(R.id.ch_viewMessageHolderText);
    textMessage.setLinkButtonActionListener(this);

    // find content view

    viewMessageContentTranslation = itemView.findViewById(R.id.ch_viewMessageContentTranslation);

    viewMessageContentAttachments = itemView.findViewById(R.id.ch_viewMessageContentAttachments);

    viewMessageContentWebPage = itemView.findViewById(R.id.ch_viewMessageContentWebPage);
    viewMessageContentWebPage.setOnWebPageActionListener(this);

    viewMessageContentReactions = itemView.findViewById(R.id.ch_viewMessageContentReactions);
    viewMessageContentReactions.setOnLongClickListener(v -> {
      onReactionsLongClicked();
      return true;
    });

    viewMessageContentGiantEmoji = itemView.findViewById(R.id.ch_viewMessageContentGiantEmoji);

    // add all contents

    contents = new ArrayList<>();

    contents.add(viewMessageContentTranslation);
    contents.add(viewMessageContentAttachments);
    contents.add(viewMessageContentWebPage);
    contents.add(viewMessageContentReactions);
    contents.add(viewMessageContentGiantEmoji);
  }

  @Override
  public void bind(Long createdAt, boolean showDate, boolean isHead, boolean isTail) {
    super.bind(createdAt, showDate, isHead, isTail);

    Stream.ofNullable(contents).forEach(content -> {
      if (content != null) {
        content.clear();
        content.resetView();
        content.setVisibility(View.GONE);
      }
    });

    if (textMessage != null) {
      textMessage.setHead(isHead);
      textMessage.initViews();
    }
  }

  // content setter

  protected void setGravity(int gravity) {
    if (viewMessageContentWebPage != null) {
      viewMessageContentWebPage.setGravity(gravity);
    }
    if (viewMessageContentAttachments != null) {
      viewMessageContentAttachments.setGravity(gravity);
    }
  }

  protected void setText(@Nullable CharSequence text) {
    if (textMessage != null) {
      if (text != null) {
        textMessage.setVisibility(View.VISIBLE);
        textMessage.setText(text);
      }
    }
  }

  protected void setBlocks(@Nullable List<Block> blocks) {
    if (textMessage != null) {
      if (blocks != null) {
        textMessage.setVisibility(View.VISIBLE);
        textMessage.setBlocks(blocks);
      }
    }
  }

  protected void setGiantEmoji(Block blocks) {
    if (viewMessageContentGiantEmoji != null) {
      viewMessageContentGiantEmoji.setVisibility(View.VISIBLE);
      viewMessageContentGiantEmoji.setText(blocks.getFormattedSpanMessage());
    }
  }

  protected void setLinkButtons(@Nullable List<Button> buttons) {
    if (textMessage != null && buttons != null && buttons.size() > 0) {
      textMessage.setVisibility(View.VISIBLE);
      textMessage.setLinkButtons(buttons);
    }
  }

  protected void setDeleteMessage() {
    if (textMessage != null) {
      textMessage.setVisibility(View.VISIBLE);
      textMessage.setDelete();
    }
  }

  protected void setWebPage(@Nullable String messageId, WebPage webPage) {
    if (viewMessageContentWebPage != null) {
      viewMessageContentWebPage.setVisibility(View.VISIBLE);
      viewMessageContentWebPage.setWebPage(messageId, webPage);
    }
  }

  protected void setAttachments(
      @Nullable String messageId,
      @Nullable List<File> attachments,
      @Nullable OnAttachmentContentActionListener listener
  ) {
    if (viewMessageContentAttachments != null) {
      viewMessageContentAttachments.resetView();

      if (messageId != null && ListUtils.hasItems(attachments)) {
        viewMessageContentAttachments.setVisibility(View.VISIBLE);
        viewMessageContentAttachments.setAttachments(messageId, attachments, listener);
      }
    }
  }

  protected void setReactions(@Nullable List<Reaction> reactions) {
    if (viewMessageContentReactions != null && reactions != null) {
      viewMessageContentReactions.setVisibility(View.VISIBLE);
      viewMessageContentReactions.setReactions(reactions);
    }
  }

  protected void setTranslation(
      @Nullable String chatId,
      @Nullable String messageId,
      String language,
      @Nullable List<Block> blocks
  ) {
    if (viewMessageContentTranslation != null) {
      viewMessageContentTranslation.setVisibility(View.VISIBLE);
      viewMessageContentTranslation.bind(chatId, messageId, language);

      // exception: attach translation binder
      translationBinder = TranslationSelector.bindTranslation(TranslationInfo.createKey(chatId, messageId, language), translationInfo -> {
        if (translationInfo != null && translationInfo.getState() == TranslationState.TRANSLATED && translationInfo.getBlocks() != null) {
          textMessage.setVisibility(View.VISIBLE);
          textMessage.setBlocks(translationInfo.getBlocks());
        } else if (blocks != null) {
          textMessage.setVisibility(View.VISIBLE);
          textMessage.setBlocks(blocks);
        }
      });
    }
  }

  @Override
  public boolean onLongClick(View v) {
    return false;
  }

  @Override
  public void onActionClick(@NonNull String actionType, @NonNull ActionButton actionButton) {
    if (this.listener != null) {
      listener.onActionClick(actionType, actionButton);

      if (getMarketing() != null) {
        this.listener.onMarketingAction(getMarketing(), null);
      }
    }
  }

  @Override
  public void onUrlClick(String url) {
    if (this.listener != null) {
      this.listener.onUrlClick(url);

      if (getMarketing() != null) {
        this.listener.onMarketingAction(getMarketing(), url);
      }
    }
  }

  @Override
  public void onWebPageClick(String url) {
    if (this.listener != null) {
      this.listener.onUrlClick(url);

      if (getMarketing() != null) {
        this.listener.onMarketingAction(getMarketing(), url);
      }
    }
  }

  @Override
  public void onWebPageLongClick() {
  }

  @Override
  public void onOpenVideoClick(File attachment, long startAt) {
    if (this.listener != null) {
      this.listener.onOpenVideoClick(attachment, startAt);

      if (getMarketing() != null) {
        this.listener.onMarketingAction(getMarketing(), null);
      }
    }
  }

  @Override
  public void onAttachmentClick(File attachment) {
    if (this.listener != null && getMarketing() != null) {
      this.listener.onMarketingAction(getMarketing(), null);
    }
  }

  @Override
  public void onAttachmentClick(File attachment, Message message) {
    if (this.listener != null) {
      this.listener.onAttachmentClick(attachment, message);

      if (getMarketing() != null) {
        this.listener.onMarketingAction(getMarketing(), null);
      }
    }
  }

  @Override
  public void onAttachmentActionClick() {
    if (this.listener != null && getMarketing() != null) {
      this.listener.onMarketingAction(getMarketing(), null);
    }
  }

  @Override
  public void onAttachmentLongClick() {
  }

  @Override
  public void onLinkButtonClick(String link) {
    if (this.listener != null) {
      if (UriUtils.isMailToUri(link)) {
        Executor.openEmail(context, link.substring(UriUtils.MAILTO_URI_PREFIX.length()));
      } else if (UriUtils.isTelUri(link)) {
        Executor.openCall(context, link.substring(UriUtils.TEL_URI_PREFIX.length()));
      } else {
        this.listener.onUrlClick(link);
      }

      if (getMarketing() != null) {
        this.listener.onMarketingAction(getMarketing(), link);
      }
    }
  }

  @Override
  public void onLinkButtonLongClick() {
  }

  @Override
  public void onMarketingAction(Marketing marketing, @Nullable String url) {
    if (this.listener != null) {
      this.listener.onMarketingAction(marketing, url);
    }
  }

  protected void onReactionsLongClicked() {
  }

  @Nullable
  protected abstract Marketing getMarketing();

  @Override
  protected void onRecycled() {
    super.onRecycled();

    Stream.ofNullable(contents).forEach(content -> {
      if (content != null) {
        content.clear();
      }
    });

    // exception: release translation binder
    if (translationBinder != null) {
      translationBinder.unbind();
      translationBinder = null;
    }
  }
}
