package com.zoyi.channel.plugin.android.activity.download;

import android.annotation.TargetApi;
import android.app.Activity;
import android.content.ContentResolver;
import android.content.ContentValues;
import android.net.Uri;
import android.os.*;
import android.provider.MediaStore;
import android.support.annotation.*;

import com.zoyi.channel.plugin.android.base.AbstractPresenter;
import com.zoyi.channel.plugin.android.global.*;
import com.zoyi.channel.plugin.android.network.*;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.okhttp3.*;
import com.zoyi.okio.*;
import com.zoyi.retrofit2.Retrofit;
import com.zoyi.retrofit2.adapter.rxjava.RxJavaCallAdapterFactory;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.functions.Func1;

import java.io.*;
import java.nio.channels.FileChannel;

public class DownloadPresenter extends AbstractPresenter<DownloadContract.View> implements DownloadContract.Presenter {

  private Activity activity;
  private DownloadContract.View view;

  @Nullable
  private Subscription downloadSubscription;

  public DownloadPresenter(DownloadContract.View view, Activity activity) {
    super(view);

    this.activity = activity;
    this.view = view;
  }

  @Override
  public void init() {
  }

  @Override
  public void download(String url, String filename, @Nullable String type) {
    if (!createDirectory(type)) {
      view.onError(ResUtils.getString("ch.common_error"));
      return;
    }

    OkHttpClient client = new OkHttpClient.Builder()
        .addNetworkInterceptor(chain -> {
          Response originResponse = chain.proceed(chain.request());

          if (originResponse.body() != null) {
            return originResponse.newBuilder()
                .body(new ProgressResponseBody(originResponse.body()))
                .build();
          }
          return originResponse;
        })
        .build();

    DownloadApi api = new Retrofit.Builder()
        .baseUrl(ServiceFactory.getRestEndPoint())
        .client(client)
        .addCallAdapterFactory(RxJavaCallAdapterFactory.create())
        .build()
        .create(DownloadApi.class);

    this.downloadSubscription = new Api<>(
        api.download(url).map((Func1<ResponseBody, Uri>) body -> handleDownload(activity, filename, type, body)))
        .run(new RestSubscriber<Uri>() {
          @Override
          public void onError(RetrofitException error) {
            view.onError(error.getMessage());
          }

          @Override
          public void onSuccess(@NonNull Uri uri) {
            view.onDownloadComplete(uri, filename, !Const.FILE_TYPE_IMAGE.equals(type) && !Const.FILE_TYPE_VIDEO.equals(type));
          }

          @Override
          public void onSuccessWithNull() {
            view.onError(ResUtils.getString("ch.common_error"));
          }
        });
  }

  @Nullable
  private Uri handleDownload(Activity activity, String filename, @Nullable String type, ResponseBody body) {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
      return applyDownloadFrom29(activity, filename, type, body);
    }
    return applyDownloadUntil29(activity, filename, body, getDownloadRootPath(type));
  }

  @RequiresApi(29)
  @Nullable
  private Uri applyDownloadFrom29(Activity activity, String filename, @Nullable String type, ResponseBody body) {
    ContentResolver contentResolver = activity.getContentResolver();
    String relativePath = getDownloadRootPath(type) + File.separator + Const.EXTERNAL_DOWNLOAD_PATH;
    ContentValues values = getValues(filename, type, body, relativePath);
    Uri uri = getUri(contentResolver, type, values);
    String pendingKey = getPendingKey(type);

    if (uri != null) {
      try {
        ParcelFileDescriptor descriptor = contentResolver.openFileDescriptor(uri, "w", null);
        if (descriptor != null) {
          FileOutputStream fos = new FileOutputStream(descriptor.getFileDescriptor());
          fos.write(body.bytes());
          fos.close();
          descriptor.close();
        }

        contentResolver.update(uri, values, null, null);

        values.clear();
        values.put(pendingKey, 0);
        contentResolver.update(uri, values, null, null);

        return uri;
      } catch (Exception ex) {
        L.e(ex.getMessage());
      }
    }
    return null;
  }

  @RequiresApi(29)
  private ContentValues getValues(String filename, @Nullable String type, ResponseBody body, String relativePath) {
    ContentValues contentValues = new ContentValues();
    MediaType contentType = body.contentType();
    String contentTypeString = contentType != null ? contentType.toString() : "";

    switch (type != null ? type : "") {
      case Const.FILE_TYPE_IMAGE:
        contentValues.put(MediaStore.Images.Media.DISPLAY_NAME, filename);
        contentValues.put(MediaStore.Images.Media.MIME_TYPE, contentTypeString);
        contentValues.put(MediaStore.Images.Media.IS_PENDING, 1);
        contentValues.put(MediaStore.Images.Media.RELATIVE_PATH, relativePath);
        break;

      case Const.FILE_TYPE_VIDEO:
        contentValues.put(MediaStore.Video.Media.DISPLAY_NAME, filename);
        contentValues.put(MediaStore.Video.Media.MIME_TYPE, contentTypeString);
        contentValues.put(MediaStore.Video.Media.IS_PENDING, 1);
        contentValues.put(MediaStore.Video.Media.RELATIVE_PATH, relativePath);
        break;

      default:
        contentValues.put(MediaStore.Downloads.DISPLAY_NAME, filename);
        contentValues.put(MediaStore.Downloads.MIME_TYPE, contentTypeString);
        contentValues.put(MediaStore.Downloads.IS_PENDING, 1);
        contentValues.put(MediaStore.Downloads.RELATIVE_PATH, relativePath);
        break;
    }

    return contentValues;
  }

  @TargetApi(29)
  private Uri getUri(ContentResolver contentResolver, @Nullable String type, ContentValues values) {
    switch (type != null ? type : "") {
      case Const.FILE_TYPE_IMAGE:
        return contentResolver.insert(MediaStore.Images.Media.getContentUri(MediaStore.VOLUME_EXTERNAL_PRIMARY), values);

      case Const.FILE_TYPE_VIDEO:
        return contentResolver.insert(MediaStore.Video.Media.getContentUri(MediaStore.VOLUME_EXTERNAL_PRIMARY), values);

      default:
        return contentResolver.insert(MediaStore.Downloads.getContentUri(MediaStore.VOLUME_EXTERNAL_PRIMARY), values);
    }
  }

  @TargetApi(29)
  private String getPendingKey(@Nullable String type) {
    switch (type != null ? type : "") {
      case Const.FILE_TYPE_IMAGE:
        return MediaStore.Images.Media.IS_PENDING;

      case Const.FILE_TYPE_VIDEO:
        return MediaStore.Video.Media.IS_PENDING;

      default:
        return MediaStore.Downloads.IS_PENDING;
    }
  }

  @Nullable
  @TargetApi(28)
  private Uri applyDownloadUntil29(Activity activity, String filename, ResponseBody body, String rootPath) {
    try {
      String root = Environment.getExternalStoragePublicDirectory(rootPath).getAbsolutePath();
      String savePath = String.format("%s/%s/%s", root, Const.EXTERNAL_DOWNLOAD_PATH, filename);
      String tempPath = String.format("%s/%s", activity.getExternalCacheDir(), RandomUtils.getRandomString(16));

      File downloadedFile = new File(tempPath);
      BufferedSink sink = Okio.buffer(Okio.sink(downloadedFile));

      sink.writeAll(body.source());
      sink.close();

      File file = new File(savePath);
      FileChannel inChannel = new FileInputStream(tempPath).getChannel();
      FileChannel outChannel = new FileOutputStream(savePath).getChannel();

      inChannel.transferTo(0, inChannel.size(), outChannel);
      inChannel.close();
      outChannel.close();

      File tempFile = new File(tempPath);
      if (tempFile.exists()) {
        tempFile.delete();
      }

      Executor.startFileMediaScan(activity, file);

      if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
        return ChannelFileProvider.getUriForFile(activity, file);
      }
      return Uri.fromFile(file);
    } catch (Exception ex) {
      return null;
    }
  }

  private String getDownloadRootPath(@Nullable String type) {
    if (type == null) {
      return Environment.DIRECTORY_DOWNLOADS;
    }

    switch (type) {
      case Const.FILE_TYPE_IMAGE:
        return Environment.DIRECTORY_PICTURES;
      case Const.FILE_TYPE_VIDEO:
        return Environment.DIRECTORY_MOVIES;
      default:
        return Environment.DIRECTORY_DOWNLOADS;
    }
  }

  private boolean createDirectory(@Nullable String type) {
    if (Build.VERSION.SDK_INT < Build.VERSION_CODES.Q) {
      try {
        File dir = new File(
            Environment.getExternalStoragePublicDirectory(getDownloadRootPath(type)),
            Const.EXTERNAL_DOWNLOAD_PATH
        );

        if (!dir.exists()) {
          if (!dir.mkdirs()) {
            throw new Exception("Directory can not created.");
          }
        }
      } catch (Exception ex) {
        return false;
      }
    }
    return true;
  }

  @Override
  public void cancel() {
    if (this.downloadSubscription != null && !this.downloadSubscription.isUnsubscribed()) {
      this.downloadSubscription.unsubscribe();
      this.downloadSubscription = null;
    }
  }

  @Override
  public void release() {
    super.release();

    cancel();
  }

  private class ProgressResponseBody extends ResponseBody {

    private ResponseBody responseBody;
    @Nullable
    private BufferedSource bufferedSource;

    public ProgressResponseBody(ResponseBody responseBody) {
      this.responseBody = responseBody;
    }

    @Override
    @Nullable
    public MediaType contentType() {
      if (responseBody != null) {
        return responseBody.contentType();
      }
      return null;
    }

    @Override
    public long contentLength() {
      if (responseBody != null) {
        return responseBody.contentLength();
      }
      return 0;
    }

    private void handleProgress(int progress) {
      activity.runOnUiThread(() -> view.onProgressUpdate(progress));
    }

    @Override
    public BufferedSource source() {
      if (this.bufferedSource == null) {
        bufferedSource = Okio.buffer(source(responseBody.source()));
      }
      return bufferedSource;
    }

    private Source source(Source source) {
      return new ForwardingSource(source) {
        private long totalBytesRead = 0;
        private int progress = 0;

        @Override
        public long read(Buffer sink, long byteCount) throws IOException {
          long bytesRead = super.read(sink, byteCount);
          totalBytesRead += bytesRead != -1 ? bytesRead : 0;

          try {
            int p = (int) (100 * totalBytesRead / contentLength());
            if (p != progress) {
              progress = p;
              handleProgress(p);
            }
          } catch (Exception ex) {
          }

          return bytesRead;
        }
      };
    }
  }
}
