package com.zoyi.channel.plugin.android.activity.lounge.view.app_messenger;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.model.rest.AppMessenger;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.Plugin;
import com.zoyi.channel.plugin.android.util.*;

import java.util.Collection;

public class LoungeIntegrationView extends ViewGroup {

  private Context context;

  private int buttonSize;
  private int spacing;

  @Nullable
  private OnIntegrationClickListener listener;

  public LoungeIntegrationView(@NonNull Context context) {
    super(context);

    init(context);
  }

  public LoungeIntegrationView(@NonNull Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);

    init(context);
  }

  public LoungeIntegrationView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);

    init(context);
  }

  private void init(Context context) {
    this.context = context;

    buttonSize = ResUtils.getDimen(context, R.dimen.ch_app_messenger_button_size);
    spacing = ResUtils.getDimen(context, R.dimen.ch_lounge_integration_button_space);
  }

  public void setListener(@Nullable OnIntegrationClickListener listener) {
    this.listener = listener;
  }

  @Override
  protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
    final int count = getChildCount();

    if (count == 0) {
      super.onMeasure(widthMeasureSpec, heightMeasureSpec);
      return;
    }

    final int availableWidth = MeasureSpec.getSize(widthMeasureSpec) - getPaddingLeft() - getPaddingRight();

    int maxColumnCount = (availableWidth + spacing) / (buttonSize + spacing);
    int maxRowCount = (count / maxColumnCount) + (count % maxColumnCount > 0 ? 1 : 0);
    int height = maxRowCount * buttonSize + (maxRowCount - 1) * spacing + getPaddingTop() + getPaddingBottom();

    int childMeasureSpec = MeasureSpec.makeMeasureSpec(buttonSize, MeasureSpec.EXACTLY);

    for (int i = 0; i < count; i++) {
      final View child = getChildAt(i);
      child.measure(childMeasureSpec, childMeasureSpec);
    }

    setMeasuredDimension(availableWidth + getPaddingLeft() + getPaddingRight(), height);
  }

  @Override
  protected void onLayout(boolean changed, int l, int t, int r, int b) {
    final int count = getChildCount();
    final int availableWidth = r - l - getPaddingLeft() - getPaddingRight();

    if (count > 0) {
      int maxColumnCount = (availableWidth + spacing) / (buttonSize + spacing);
      int maxRowCount = (count / maxColumnCount) + (count % maxColumnCount > 0 ? 1 : 0);
      int baseColumnCount = count / maxRowCount;
      int restCount = count % maxRowCount;

      int viewCount = 0;

      for (int i = 1; i <= maxRowCount; i++) {
        int columnCount = baseColumnCount + (restCount >= i ? 1 : 0);
        int startX = (availableWidth - (columnCount * buttonSize) - ((columnCount - 1) * spacing)) / 2 + getPaddingLeft();
        int startY = (i - 1) * (buttonSize + spacing) + getPaddingTop();

        for (int j = 0; j < columnCount; j++) {
          final View child = getChildAt(j + viewCount);

          child.layout(
              startX + j * (buttonSize + spacing),
              startY,
              startX + j * (buttonSize + spacing) + buttonSize,
              startY + buttonSize
          );
        }

        viewCount += columnCount;
      }
    }
  }

  public void addAppMessengers(Channel channel, Plugin plugin, Collection<AppMessenger> appMessengers) {
    removeAllViews();
    int activeCount = appMessengers.size();

    // App
    for (AppMessenger appMessenger : appMessengers) {
      this.addView(new MessengerAppButtonView(context, appMessenger, listener));
    }

    // default link
    if (channel != null &&
        plugin != null &&
        channel.getDomain() != null &&
        channel.getDefaultPluginId() != null &&
        channel.getDefaultPluginId().equals(plugin.getId())
    ) {
      this.addView(new LinkAppButtonView(context, String.format("https://%s.channel.io", channel.getDomain()), listener));
      activeCount++;
    }

    // phone
    if (channel != null && channel.getPhoneNumber() != null && Executor.canCall(context)) {
      this.addView(new CallAppButtonView(context, channel.getPhoneNumber(), listener));
      activeCount++;
    }

    Views.setVisibility(this, activeCount > 0);
  }
}
