package com.zoyi.channel.plugin.android.view.youtube.ui;

import android.content.Intent;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.support.annotation.Nullable;
import android.support.v4.content.ContextCompat;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.util.Initializer;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.view.external.cpv.CircularProgressView;
import com.zoyi.channel.plugin.android.view.youtube.player.PlayerConstants;
import com.zoyi.channel.plugin.android.view.youtube.player.YouTubePlayer;
import com.zoyi.channel.plugin.android.view.youtube.player.listener.YouTubePlayerFullScreenListener;
import com.zoyi.channel.plugin.android.view.youtube.player.listener.YouTubePlayerListener;
import com.zoyi.channel.plugin.android.view.youtube.player.views.LegacyYouTubePlayerView;
import com.zoyi.channel.plugin.android.view.youtube.ui.menu.YouTubePlayerMenu;
import com.zoyi.channel.plugin.android.view.youtube.ui.menu.defaultMenu.DefaultYouTubePlayerMenu;
import com.zoyi.channel.plugin.android.view.youtube.ui.utils.FadeViewHelper;
import com.zoyi.channel.plugin.android.view.youtube.ui.views.CHYouTubePlayerSeekBar;

public class DefaultPlayerUiController implements PlayerUiController, YouTubePlayerListener, YouTubePlayerFullScreenListener,
    CHYouTubePlayerSeekBar.YouTubePlayerSeekBarListener {

  private YouTubePlayerMenu youTubePlayerMenu;

  private YouTubePlayer youTubePlayer;

  /**
   * View used for for intercepting clicks and for drawing a black background. Could have used controlsContainer, but in this way I'm able
   * to hide all the control at once by hiding controlsContainer
   */
  private View panel;

  private View controlsContainer;
  private LinearLayout extraViewsContainer;

  private TextView videoTitle;
  private TextView liveVideoIndicator;

  private CircularProgressView progressBar;
  private ImageView menuButton;
  private ImageView playPauseButton;
  private ImageView youTubeButton;
  private ImageView fullScreenButton;

  private ImageView customActionLeft;
  private ImageView customActionRight;

  private CHYouTubePlayerSeekBar youtubePlayerSeekBar;

  private View.OnClickListener onFullScreenButtonListener;
  private View.OnClickListener onMenuButtonClickListener;

  private FadeViewHelper fadeControlsContainer;

  private boolean isPlaying = false;
  private boolean isPlayPauseButtonEnabled = true;
  private boolean isCustomActionLeftEnabled = false;
  private boolean isCustomActionRightEnabled = false;

  public DefaultPlayerUiController(LegacyYouTubePlayerView youTubePlayerView, YouTubePlayer youTubePlayer) {
    init(youTubePlayerView, youTubePlayer);
  }

  @Initializer
  private void init(LegacyYouTubePlayerView youTubePlayerView, YouTubePlayer youTubePlayer) {
    View controlsView = View.inflate(youTubePlayerView.getContext(), R.layout.ch_plugin_ayp_default_player_ui, youTubePlayerView);

    this.youTubePlayer = youTubePlayer;

    youTubePlayerMenu = new DefaultYouTubePlayerMenu(youTubePlayerView.getContext());

    panel = controlsView.findViewById(R.id.panel);
    controlsContainer = controlsView.findViewById(R.id.ch_plugin_controls_container);
    extraViewsContainer = controlsView.findViewById(R.id.ch_plugin_extra_views_container);

    videoTitle = controlsView.findViewById(R.id.ch_plugin_video_title);
    liveVideoIndicator = controlsView.findViewById(R.id.ch_plugin_live_video_indicator);

    progressBar = controlsView.findViewById(R.id.ch_plugin_progress);
    menuButton = controlsView.findViewById(R.id.ch_plugin_menu_button);
    playPauseButton = controlsView.findViewById(R.id.ch_plugin_play_pause_button);
    youTubeButton = controlsView.findViewById(R.id.ch_plugin_youtube_button);
    fullScreenButton = controlsView.findViewById(R.id.ch_plugin_fullscreen_button);

    customActionLeft = controlsView.findViewById(R.id.ch_plugin_custom_action_left_button);
    customActionRight = controlsView.findViewById(R.id.ch_plugin_custom_action_right_button);

    youtubePlayerSeekBar = controlsView.findViewById(R.id.ch_plugin_youtube_player_seekbar);

    fadeControlsContainer = new FadeViewHelper(controlsContainer);

    onFullScreenButtonListener = view -> youTubePlayerView.toggleFullScreen();
    onMenuButtonClickListener = view -> youTubePlayerMenu.show(menuButton);

    initClickListeners();
  }


  private void initClickListeners() {
    youTubePlayer.addListener(youtubePlayerSeekBar);
    youTubePlayer.addListener(fadeControlsContainer);

    youtubePlayerSeekBar.setYoutubePlayerSeekBarListener(this);
    panel.setOnClickListener(view -> fadeControlsContainer.toggleVisibility());
    playPauseButton.setOnClickListener(view -> onPlayButtonPressed());
    fullScreenButton.setOnClickListener(view -> onFullScreenButtonListener.onClick(fullScreenButton));
    menuButton.setOnClickListener(view -> onMenuButtonClickListener.onClick(menuButton));
  }

  @Override
  public PlayerUiController showVideoTitle(boolean show) {
    videoTitle.setVisibility(show ? View.VISIBLE : View.GONE);
    return this;
  }

  @Override
  public PlayerUiController setVideoTitle(String videoTitle) {
    this.videoTitle.setText(videoTitle);
    return this;
  }

  @Override
  public PlayerUiController showUi(boolean show) {
    fadeControlsContainer.setDisabled(!show);
    controlsContainer.setVisibility(show ? View.VISIBLE : View.INVISIBLE);
    return this;
  }

  @Override
  public PlayerUiController showPlayPauseButton(boolean show) {
    playPauseButton.setVisibility(show ? View.VISIBLE : View.GONE);

    isPlayPauseButtonEnabled = show;
    return this;
  }

  @Override
  public PlayerUiController enableLiveVideoUi(boolean enable) {
    youtubePlayerSeekBar.setVisibility(enable ? View.INVISIBLE : View.VISIBLE);
    liveVideoIndicator.setVisibility(enable ? View.VISIBLE : View.GONE);
    return this;
  }

  @Override
  public PlayerUiController setCustomAction1(Drawable icon, @Nullable View.OnClickListener clickListener) {
    customActionLeft.setImageDrawable(icon);
    customActionLeft.setOnClickListener(clickListener);
    showCustomAction1(true);
    return this;
  }

  @Override
  public PlayerUiController setCustomAction2(Drawable icon, @Nullable View.OnClickListener clickListener) {
    customActionRight.setImageDrawable(icon);
    customActionRight.setOnClickListener(clickListener);
    showCustomAction2(true);
    return this;
  }

  @Override
  public PlayerUiController showCustomAction1(boolean show) {
    isCustomActionLeftEnabled = show;
    customActionLeft.setVisibility(show ? View.VISIBLE : View.GONE);
    return this;
  }

  @Override
  public PlayerUiController showCustomAction2(boolean show) {
    isCustomActionRightEnabled = show;
    customActionRight.setVisibility(show ? View.VISIBLE : View.GONE);
    return this;
  }

  @Override
  public PlayerUiController showMenuButton(boolean show) {
    menuButton.setVisibility(show ? View.VISIBLE : View.GONE);
    return this;
  }

  @Override
  public PlayerUiController setMenuButtonClickListener(View.OnClickListener customMenuButtonClickListener) {
    onMenuButtonClickListener = customMenuButtonClickListener;
    return this;
  }

  @Override
  public PlayerUiController showCurrentTime(boolean show) {
    youtubePlayerSeekBar.setVisibility(show ? View.VISIBLE : View.GONE);
    return this;
  }

  @Override
  public PlayerUiController showDuration(boolean show) {
    youtubePlayerSeekBar.getVideoDurationTextView().setVisibility(show ? View.VISIBLE : View.GONE);
    return this;
  }

  @Override
  public PlayerUiController showSeekBar(boolean show) {
    youtubePlayerSeekBar.getSeekBar().setVisibility(show ? View.VISIBLE : View.INVISIBLE);
    return this;
  }

  @Override
  public PlayerUiController showFullscreenButton(boolean show) {
    fullScreenButton.setVisibility(show ? View.VISIBLE : View.GONE);
    return this;
  }

  @Override
  public PlayerUiController setFullScreenButtonClickListener(View.OnClickListener customFullScreenButtonClickListener) {
    onFullScreenButtonListener = customFullScreenButtonClickListener;
    return this;
  }

  @Override
  public PlayerUiController showBufferingProgress(boolean show) {
    youtubePlayerSeekBar.setShowBufferingProgress(show);
    return this;
  }

  @Override
  public PlayerUiController showYouTubeButton(boolean show) {
    youTubeButton.setVisibility(show ? View.VISIBLE : View.GONE);
    return this;
  }

  @Override
  public PlayerUiController addView(View view) {
    extraViewsContainer.addView(view, 0);
    return this;
  }

  @Override
  public PlayerUiController removeView(View view) {
    extraViewsContainer.removeView(view);
    return this;
  }

  @Nullable
  @Override
  public YouTubePlayerMenu getMenu() {
    return youTubePlayerMenu;
  }

  private void onPlayButtonPressed() {
    if (isPlaying) {
      youTubePlayer.pause();
    } else {
      youTubePlayer.play();
    }
  }

  @Override
  public void onYouTubePlayerEnterFullScreen() {
    fullScreenButton.setImageResource(R.drawable.ch_plugin_ayp_ic_fullscreen_exit_24dp);
  }

  @Override
  public void onYouTubePlayerExitFullScreen() {
    fullScreenButton.setImageResource(R.drawable.ch_plugin_ayp_ic_fullscreen_24dp);
  }

  private void updateState(PlayerConstants.PlayerState state) {
    switch (state) {
      case ENDED:
      case PAUSED:
        isPlaying = false;
        break;
      case PLAYING:
        isPlaying = true;
        break;
      default:
        break;
    }

    updatePlayPauseButtonIcon(!isPlaying);
  }

  private void updatePlayPauseButtonIcon(boolean playing) {
    playPauseButton.setImageResource(playing ? R.drawable.ch_plugin_ayp_ic_pause_36dp : R.drawable.ch_plugin_ayp_ic_play_36dp);
  }

  @Override
  public void seekTo(float time) {
    youTubePlayer.seekTo(time);
  }

  // YouTubePlayer callbacks

  @Override
  public void onStateChange(YouTubePlayer youTubePlayer, PlayerConstants.PlayerState state) {
    updateState(state);

    if (
        state == PlayerConstants.PlayerState.PLAYING
            || state == PlayerConstants.PlayerState.PAUSED
            || state == PlayerConstants.PlayerState.VIDEO_CUED
    ) {
      panel.setBackgroundColor(ContextCompat.getColor(panel.getContext(), android.R.color.transparent));
      progressBar.setVisibility(View.GONE);

      if (isPlayPauseButtonEnabled) {
        playPauseButton.setVisibility(View.VISIBLE);
      }
      if (isCustomActionLeftEnabled) {
        customActionLeft.setVisibility(View.VISIBLE);
      }
      if (isCustomActionRightEnabled) {
        customActionRight.setVisibility(View.VISIBLE);
      }

      updatePlayPauseButtonIcon(state == PlayerConstants.PlayerState.PLAYING);

    } else {
      updatePlayPauseButtonIcon(false);

      if (state == PlayerConstants.PlayerState.BUFFERING) {
        progressBar.setVisibility(View.VISIBLE);
        panel.setBackgroundColor(ContextCompat.getColor(panel.getContext(), android.R.color.transparent));
        if (isPlayPauseButtonEnabled) {
          playPauseButton.setVisibility(View.INVISIBLE);
        }

        customActionLeft.setVisibility(View.GONE);
        customActionRight.setVisibility(View.GONE);
      }

      if (state == PlayerConstants.PlayerState.UNSTARTED) {
        progressBar.setVisibility(View.GONE);
        if (isPlayPauseButtonEnabled) {
          playPauseButton.setVisibility(View.VISIBLE);
        }
      }
    }
  }

  @Override
  public void onVideoId(YouTubePlayer youTubePlayer, String videoId) {
    youTubeButton.setOnClickListener(view -> {
      Intent intent = new Intent(
          Intent.ACTION_VIEW,
          Uri.parse("http://www.youtube.com/watch?v=" + videoId + "#t=" + youtubePlayerSeekBar.getSeekBar().getProgress())
      );
      try {
        youTubeButton.getContext().startActivity(intent);
      } catch (Exception e) {
        L.e(e.getMessage() == null ? "Can't open url to YouTube" : e.getMessage());
      }
    });
  }

  @Override
  public void onReady(YouTubePlayer youTubePlayer) {

  }

  @Override
  public void onPlaybackQualityChange(YouTubePlayer youTubePlayer, PlayerConstants.PlaybackQuality playbackQuality) {

  }

  @Override
  public void onPlaybackRateChange(YouTubePlayer youTubePlayer, PlayerConstants.PlaybackRate playbackRate) {

  }

  @Override
  public void onError(YouTubePlayer youTubePlayer, PlayerConstants.PlayerError error) {

  }

  @Override
  public void onCurrentSecond(YouTubePlayer youTubePlayer, float second) {

  }

  @Override
  public void onVideoDuration(YouTubePlayer youTubePlayer, float duration) {

  }

  @Override
  public void onVideoLoadedFraction(YouTubePlayer youTubePlayer, float loadedFraction) {

  }

  @Override
  public void onApiChange(YouTubePlayer youTubePlayer) {

  }
}
