package com.zoyi.channel.plugin.android;

import android.app.*;
import android.content.Context;
import android.os.Build;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.widget.FrameLayout;

import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.store.GlobalStore;
import com.zoyi.channel.plugin.android.store.TimerStore;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.HashSet;
import java.util.concurrent.TimeUnit;

public class ActivityLifecycleManager implements Application.ActivityLifecycleCallbacks {

  private int running = 0;

  @Nullable
  private Subscription workingTimer;

  private HashSet<String> unmanagedActivities = new HashSet<>();

  @Override
  public void onActivityCreated(Activity activity, Bundle savedInstanceState) {
  }

  @Override
  public void onActivityStarted(Activity activity) {
    if (isActivityShowWhenLocked(activity)) {
      L.d("ActivityLifecycle:Started:Locked " + activity.getClass().getSimpleName());

      unmanagedActivities.add(getActivityHash(activity));
    } else {
      L.d("ActivityLifecycle:Started:Unlocked " + activity.getClass().getSimpleName());

      if (!ChannelUtils.isChannelPluginActivity(activity)) {
        FrameLayout root = activity.findViewById(R.id.channel_io_view);
        if (root == null && !activity.getClass().isAnnotationPresent(SkipAttachChannelView.class)) {
          root = new FrameLayout(activity);
          root.setClipChildren(false);
          root.setClipToPadding(false);
          root.setFitsSystemWindows(true);
          root.setId(R.id.channel_io_view);

          try {
            root.addView(new ChannelView(activity));

            activity.addContentView(root, new FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.MATCH_PARENT,
                FrameLayout.LayoutParams.MATCH_PARENT));
          } catch (Exception e) {
            L.e(e.getMessage());
          }
        }
      }

      GlobalStore.get().topActivity.set(activity);

      if (++running == 1 && GlobalStore.get().bootState.get()) {
        SocketManager.connect();
        attachTimestampTimer();
      }
    }
  }

  @Override
  public void onActivityResumed(Activity activity) {
    L.d("ActivityLifecycle:Resumed: " + activity.getClass().getSimpleName());
    GlobalStore.get().topActivity.set(activity);
  }

  @Override
  public void onActivityPaused(Activity activity) {
    L.d("ActivityLifecycle:Paused: " + activity.getClass().getSimpleName());

    if (GlobalStore.get().topActivity.get() == activity) {
      GlobalStore.get().topActivity.set(null);
    }
  }

  @Override
  public void onActivityStopped(Activity activity) {
    if (unmanagedActivities.contains(getActivityHash(activity))) {
      L.d("ActivityLifecycle:Stopped:Locked " + activity.getClass().getSimpleName() + " " + running);

      unmanagedActivities.remove(getActivityHash(activity));
    } else {
      L.d("ActivityLifecycle:Stopped:Unlocked " + activity.getClass().getSimpleName() + " " + running + " -> " + (running - 1));

      if (GlobalStore.get().topActivity.get() == activity) {
        GlobalStore.get().topActivity.set(null);
      }

      if (--running == 0) {
        SocketManager.disconnect();
        detachTimestampTimer();
      }
    }
  }

  @Override
  public void onActivitySaveInstanceState(Activity activity, Bundle outState) {
  }

  @Override
  public void onActivityDestroyed(Activity activity) {
  }

  public void attachTimestampTimer() {
    detachTimestampTimer();

    workingTimer = Observable.interval(60, TimeUnit.SECONDS)
        .onBackpressureBuffer()
        .subscribeOn(Schedulers.io())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(l -> TimerStore.get().now.set(TimeUtils.getCurrentTime()));
  }

  public void detachTimestampTimer() {
    if (workingTimer != null) {
      if (!workingTimer.isUnsubscribed()) {
        workingTimer.unsubscribe();
      }
      workingTimer = null;
    }
  }

  private boolean isActivityShowWhenLocked(Activity activity) {
    KeyguardManager keyguardManager = (KeyguardManager) activity.getSystemService(Context.KEYGUARD_SERVICE);

    if (keyguardManager != null) {
      if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP_MR1) {
        return keyguardManager.isKeyguardLocked();
      } else {
        return keyguardManager.inKeyguardRestrictedInputMode();
      }
    }

    return false;
  }

  private String getActivityHash(Activity activity) {
    return String.format("%s:%s", activity.getClass().getSimpleName(), activity.hashCode());
  }
}
