package com.zoyi.channel.plugin.android.push;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Bitmap;
import android.os.Bundle;
import android.provider.Settings;
import android.support.annotation.Nullable;
import android.support.v4.app.NotificationCompat;
import android.support.v4.app.NotificationManagerCompat;
import android.text.TextUtils;

import com.zoyi.channel.plugin.android.global.*;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.store.SettingsStore;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.com.bumptech.glide.Glide;
import com.zoyi.com.bumptech.glide.request.target.Target;
import com.zoyi.okhttp3.RequestBody;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.Map;
import java.util.concurrent.TimeUnit;

import static com.zoyi.channel.plugin.android.global.Const.FORMAT_PUSH_TOKENS_KEY;

public class ChannelPushManager {

  private static final int NOTIFICATION_ID = Integer.MAX_VALUE / 2;

  private static Bundle convertMessageMapToBundle(Map<String, String> message) {
    Bundle bundle = new Bundle();

    for (Map.Entry<String, String> entry : message.entrySet()) {
      bundle.putString(entry.getKey(), entry.getValue());
    }
    return bundle;
  }

  public static boolean isChannelPushNotification(Map<String, String> message) {
    return isChannelPushNotification(convertMessageMapToBundle(message));
  }

  public static boolean isChannelPushNotification(Bundle message) {
    if (message == null) {
      return false;
    }
    PushData pushData = new PushData(message);

    return pushData.isChannelPluginMessage();
  }

  public static void sendTokenToChannelIO(@Nullable final Context context) {
    if (context != null) {
      String token = PrefSupervisor.getDeviceToken(context);
      String sentToken = PrefSupervisor.getSentDeviceToken(context);

      if (!TextUtils.isEmpty(token) && !CompareUtils.isSame(token, sentToken)) {
        PrefSupervisor.setSentDeviceToken(context, token);

        RequestBody body = RequestUtils.form()
            .set("key", String.format(FORMAT_PUSH_TOKENS_KEY, getDeviceId(context)))
            .set("token", token)
            .create();

        Api.getApi()
            .sendToken(body)
            .subscribeOn(Schedulers.newThread())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(new RestSubscriber<Void>());
      }
    }
  }

  @SuppressLint("HardwareIds")
  public static void deleteToken(Context context, @Nullable String jwt) {
    if (jwt != null) {
      Api.getApi()
          .deleteToken(SettingsStore.get().locale.get().toString(), String.format(FORMAT_PUSH_TOKENS_KEY, getDeviceId(context)), jwt)
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<Void>());
    }

    PrefSupervisor.setSentDeviceToken(context, null);
  }

  @SuppressLint("HardwareIds")
  private static String getDeviceId(Context context) {
    String key = PrefSupervisor.getDeviceId(context);

    if (TextUtils.isEmpty(key)) {
      key = Settings.Secure.getString(context.getContentResolver(), Settings.Secure.ANDROID_ID);
      PrefSupervisor.setDeviceId(context, key);
    }
    return key;
  }

  public static void showPushNotification(Context context, Map<String, String> message) {
    showPushNotification(context, convertMessageMapToBundle(message));
  }

  public static void showPushNotification(Context context, Bundle message) {
    if (PrefSupervisor.hasBootFlag(context)) {
      String personId = message.getString("personId");
      String personType = message.getString("personType");

      if (CompareUtils.isSame(personType, Const.USER) &&
          CompareUtils.exists(personId, PrefSupervisor.getUserId(context))) {

        sendNotification(context, message);

        if (message.getString("chatId") != null) {
          receivePushNotification(message.getString("chatId"));
        }
      }
    }
  }

  private static void sendNotification(Context context, Bundle message) {
    NotificationCompat.Builder builder = NotificationFactory.createPushMessage(context, new PushData(message));

    if (builder != null) {
      try {
        String avatarUrl = message.getString("avatarUrl");
        if (avatarUrl == null) {
          throw new Exception("avatarUrl is null");
        }

        Bitmap avatarBitmap = Glide.with(context)
            .load(avatarUrl)
            .asBitmap()
            .into(Target.SIZE_ORIGINAL, Target.SIZE_ORIGINAL)
            .get(3000, TimeUnit.MILLISECONDS);

        if (avatarBitmap == null) {
          throw new Exception("avatar bitmap is null");
        }

        builder.setLargeIcon(ImageUtils.getCircularBitmap(avatarBitmap));
      } catch (Exception ex) {
        L.e(ex.getMessage());
      }

      NotificationManagerCompat.from(context).notify(NOTIFICATION_ID, builder.build());
    }
  }

  private static void receivePushNotification(String chatId) {
    Api.getApi()
        .receivePushNotification(chatId)
        .onBackpressureBuffer()
        .subscribeOn(Schedulers.io())
        .subscribe(new RestSubscriber<Void>());
  }

  static void clearNotification(Context context) {
    try {
      NotificationManagerCompat.from(context).cancel(NOTIFICATION_ID);
    } catch (Exception ex) {
      L.e(ex.getMessage());
    }
  }
}
