package com.zoyi.channel.plugin.android.activity.chat3;

import android.content.Intent;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.content.ContextCompat;
import android.support.v7.app.AlertDialog;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.view.View;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.base.BaseActivity2;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageContentClickListener;
import com.zoyi.channel.plugin.android.activity.chat.model.SendingMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.view.NewMessageAlertView;
import com.zoyi.channel.plugin.android.activity.chat3.contract.ChatContract;
import com.zoyi.channel.plugin.android.activity.chat3.listener.MessageActionListener;
import com.zoyi.channel.plugin.android.activity.chat3.listener.OnProfileUpdateRequestListener;
import com.zoyi.channel.plugin.android.activity.chat3.model.ProfileBotMessageItem;
import com.zoyi.channel.plugin.android.activity.chat3.navigation.HostHeaderView;
import com.zoyi.channel.plugin.android.activity.chat3.view.ChatInputView;
import com.zoyi.channel.plugin.android.activity.common.chat.ChatContentType;
import com.zoyi.channel.plugin.android.activity.common.error.ErrorRefreshView;
import com.zoyi.channel.plugin.android.activity.common.navigation.ChannelInfoNavigationView;
import com.zoyi.channel.plugin.android.activity.common.navigation.NavigationView;
import com.zoyi.channel.plugin.android.activity.download.DownloadActivity;
import com.zoyi.channel.plugin.android.activity.photo_album.PhotoAlbumActivity;
import com.zoyi.channel.plugin.android.activity.photo_picker.PhotoPickerActivity;
import com.zoyi.channel.plugin.android.activity.video.VideoViewerActivity;
import com.zoyi.channel.plugin.android.enumerate.ActionType;
import com.zoyi.channel.plugin.android.enumerate.FetchState;
import com.zoyi.channel.plugin.android.enumerate.Transition;
import com.zoyi.channel.plugin.android.global.Action;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.ActionButton;
import com.zoyi.channel.plugin.android.model.entity.Previewable;
import com.zoyi.channel.plugin.android.model.entity.ProfileEntity;
import com.zoyi.channel.plugin.android.model.rest.Country;
import com.zoyi.channel.plugin.android.model.rest.File;
import com.zoyi.channel.plugin.android.model.rest.Message;
import com.zoyi.channel.plugin.android.model.rest.UserChat;
import com.zoyi.channel.plugin.android.store2.GlobalStore;
import com.zoyi.channel.plugin.android.store2.ChatStore;
import com.zoyi.channel.plugin.android.store2.ProfileBotStore;
import com.zoyi.channel.plugin.android.store2.TranslationStore;
import com.zoyi.channel.plugin.android.util.Executor;
import com.zoyi.channel.plugin.android.util.Initializer;
import com.zoyi.channel.plugin.android.util.IntentUtils;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.channel.plugin.android.util.io.Keyboard;
import com.zoyi.channel.plugin.android.util.message_format.type.LinkType;
import com.zoyi.channel.plugin.android.view.button.ChatStartButton;
import com.zoyi.channel.plugin.android.view.dialog.CountryCodeDialog;
import com.zoyi.channel.plugin.android.view.handler.InfiniteScrollListener;
import com.zoyi.channel.plugin.android.view.layout.PushBotSaveButton;
import com.zoyi.channel.plugin.android.view.scrollview.RecyclerBottomPlaceholderLayout;

import java.util.List;

public class ChatActivity
    extends BaseActivity2
    implements ChatContract.View, OnMessageContentClickListener, MessageActionListener, OnProfileUpdateRequestListener {

  private NavigationView navigation;
  private ChannelInfoNavigationView channelHeader;
  private HostHeaderView hostHeader;

  private View layoutContent;
  private View layoutProgress;
  private ErrorRefreshView layoutError;

  private ChatInputView inputView;

  private RecyclerView recyclerView;

  private NewMessageAlertView newMessageAlertView;

  private RecyclerBottomPlaceholderLayout bottomPlaceholder;
  private PushBotSaveButton buttonPushBotSave;
  private ChatStartButton buttonNewChat;

  private ChatAdapter adapter;
  private LinearLayoutManager layoutManager;

  private ChatContract.Presenter presenter;

  @Nullable
  private KeyboardUtils keyboardUtils;

  @Initializer
  @Override
  protected boolean onCreate() {
    init(R.layout.ch_plugin_activity_chat3);

    // intent initialize

    Intent intent = getIntent();
    ChatContentType contentType = ChatContentType.NONE;
    String contentId = null;

    if (intent != null) {
      contentType = ChatContentType.fromString(intent.getStringExtra(Const.EXTRA_CHAT_CONTENT_TYPE));
      contentId = intent.getStringExtra(Const.EXTRA_CHAT_CONTENT_ID);
    }

    // view initialize

    navigation = findViewById(R.id.navigationChat);
    navigation.addAction(R.drawable.ch_plugin_close_white, v -> Action.invoke(ActionType.EXIT));

    channelHeader = findViewById(R.id.navigationChatChannel);
    hostHeader = findViewById(R.id.navigationChatHost);
    hostHeader.setVisibility(View.GONE);

    layoutContent = findViewById(R.id.layoutChatContent);
    layoutProgress = findViewById(R.id.layoutChatProgress);
    layoutError = findViewById(R.id.layoutChatError);

    recyclerView = findViewById(R.id.recyclerViewChat);

    newMessageAlertView = findViewById(R.id.viewNewMessageAlert);
    newMessageAlertView.setListener(() -> {
      scrollToBottom();
      newMessageAlertView.hide();
    });

    inputView = findViewById(R.id.viewChatInput);
    inputView.setMessageActionListener(this);

    // adapter initialize

    adapter = new ChatAdapter();
    adapter.setOnMessageContentClickListener(this);
    adapter.setOnProfileUpdateRequestListener(this);

    // presenter initialize

    presenter = new ChatPresenter(this, adapter, adapter, contentType, contentId);

    // init recycler view

    layoutManager = new LinearLayoutManager(this);

    recyclerView.setLayoutManager(layoutManager);
    recyclerView.setAdapter(adapter);
    recyclerView.setItemAnimator(null);
    recyclerView.addOnScrollListener(new InfiniteScrollListener(
        layoutManager,
        InfiniteScrollListener.TOP,
        recyclerView
    ) {
      @Override
      public void scrolledInList() {
      }

      @Override
      public void scrollAttachedToBottom() {
        newMessageAlertView.hide();
      }

      @Override
      public void refresh() {
        presenter.fetchBackwardMessages();
      }
    });

    // bottom placeholder

    bottomPlaceholder = findViewById(R.id.bottomLayoutChat);
    bottomPlaceholder.setRecyclerView(recyclerView);
    bottomPlaceholder.setStackFromEnd(true);

    buttonPushBotSave = findViewById(R.id.buttonChatSavePushBot);
    buttonPushBotSave.setClickListener(v -> presenter.createPushBotChat());

    buttonNewChat = findViewById(R.id.buttonChatStartNewChat);
    buttonNewChat.setOnClickListener(v -> finish(Const.RESULT_CHAT_OPEN_NEW_CHAT, Transition.NONE));

    // keyboard adjust resize utils

    keyboardUtils = new KeyboardUtils(this, findViewById(R.id.rootChatCoordinatorView), recyclerView);
    keyboardUtils.enable();

    bindPresenter(presenter);

    GlobalStore.get().activeChatActivity.set(this);

    return true;
  }

  @Override
  protected void onPause() {
    super.onPause();

    if (isFinishing()) {
      ChatStore.get().reset();
      TranslationStore.get().reset();
      ProfileBotStore.get().reset();

      ChatActivity chatActivity = GlobalStore.get().activeChatActivity.get();

      if (this == chatActivity) {
        GlobalStore.get().activeChatActivity.set(null);
      }

      Action.invoke(ActionType.CHAT_CLOSED);
    }
  }

  @Override
  protected void onStop() {
    super.onStop();

    KeyboardUtils.hideKeyboard(this);
  }

  @Override
  protected void onDestroy() {
    super.onDestroy();

    if (keyboardUtils != null) {
      keyboardUtils.disable();
      keyboardUtils = null;
    }
  }

  @Override
  public void onFetchStateChange(FetchState fetchState) {
    layoutContent.setVisibility(View.GONE);
    layoutProgress.setVisibility(View.GONE);
    layoutError.setVisibility(View.GONE);

    switch (fetchState) {
      case LOADING:
        layoutProgress.setVisibility(View.VISIBLE);
        break;

      case FAILED:
        layoutError.setVisibility(View.VISIBLE);
        break;

      case COMPLETE:
        layoutContent.setVisibility(View.VISIBLE);
        break;
    }
  }

  @Override
  public void onWelcomeStateChange(ChatContentType contentType, boolean acceptInput) {
    channelHeader.setVisibility(View.VISIBLE);
    hostHeader.setVisibility(View.GONE);

    // input & bottom placeholder
    // input view should set visibility case by case to keep focus

    buttonPushBotSave.setVisibility(View.GONE);
    buttonNewChat.setVisibility(View.GONE);

    switch (contentType) {
      case NONE:
        inputView.setVisibility(View.VISIBLE);
        inputView.setEnabled(acceptInput);
        break;

      case SUPPORT_BOT_CHAT:
        inputView.setVisibility(View.GONE);
        break;

      case PUSH_BOT_CHAT:
        inputView.setVisibility(View.GONE);
        buttonPushBotSave.setVisibility(View.VISIBLE);
        break;
    }
  }

  @Override
  public void setPushBotSaveButtonVisibility(boolean visible) {
    buttonPushBotSave.setVisibility(visible ? View.VISIBLE : View.GONE);
  }

  @Override
  public void onChatStateChange(UserChat userChat, boolean acceptInput, boolean temporaryInputOpened) {
    if (userChat.isStateRemoved()) {
      finish(Transition.NONE);
      return;
    }

    inputView.initUserChat(userChat.getId());

    // navigation

    if (Const.MANAGER.equals(userChat.getAssigneeType()) && userChat.getAssigneeId() != null) {
      channelHeader.setVisibility(View.GONE);
      hostHeader.setVisibility(View.VISIBLE);
      hostHeader.attachUserChat(userChat.getId());
    } else {
      channelHeader.setVisibility(View.VISIBLE);
      hostHeader.setVisibility(View.GONE);
    }

    // input & bottom placeholder
    // input view should set visibility case by case to keep focus

    buttonPushBotSave.setVisibility(View.GONE);

    if (userChat.isSupporting()) {
      Keyboard.close(this, inputView.getEditText());

      inputView.setVisibility(View.GONE);
      buttonNewChat.setVisibility(View.GONE);
    } else if (userChat.isStateClosed()) {
      Keyboard.close(this, inputView.getEditText());

      inputView.setVisibility(View.GONE);
      buttonNewChat.setVisibility(View.VISIBLE);
    } else if (userChat.isStateSolved()) {
      if (temporaryInputOpened) {
        inputView.setVisibility(View.VISIBLE);
        inputView.setEnabled(acceptInput);
        buttonNewChat.setVisibility(View.GONE);
      } else {
        Keyboard.close(this, inputView.getEditText());

        inputView.setVisibility(View.GONE);
        buttonNewChat.setVisibility(View.GONE);
      }
    } else {
      inputView.setVisibility(View.VISIBLE);
      inputView.setEnabled(acceptInput);
      buttonNewChat.setVisibility(View.GONE);
    }
  }

  @Override
  public boolean isScrollOnBottom() {
    return !recyclerView.canScrollVertically(1);
  }

  @Override
  public void scrollToBottom() {
    layoutManager.scrollToPosition(adapter.getItemCount() - 1);
  }

  @Override
  public void showNewMessageAlert(ProfileEntity profileEntity) {
    newMessageAlertView.show(profileEntity);
  }

  // Handle message content action

  @Override
  public void onTranslationButtonClick(Message message) {
    presenter.translate(message);
  }

  @Override
  public void onImageClick(File file) {
    if (file != null) {
      IntentUtils.setNextActivity(this, PhotoAlbumActivity.class)
          .putExtra(Const.EXTRA_FILE_ID, file.getId())
          .startActivityForResult(Const.REQUEST_PHOTO_ALBUM);
    }
  }

  @Override
  public void onWebPageClick(String url) {
    if (url != null) {
      Executor.executeLinkAction(this, url, LinkType.URL);
    }
  }

  @Override
  public void onFileClick(File file) {
    if (file != null) {
      if (file.getType() != null && file.getType().startsWith("video")) {
        IntentUtils.setNextActivity(this, VideoViewerActivity.class)
            .putExtra(Const.EXTRA_FILE_NAME, file.getName())
            .putExtra(Const.EXTRA_URL, file.getUrl())
            .startActivity();
      } else {
        IntentUtils.setNextActivity(this, DownloadActivity.class)
            .putExtra(Const.EXTRA_URL, file.getUrl())
            .putExtra(Const.EXTRA_FILE_NAME, file.getFilename())
            .putExtra(Const.EXTRA_IMAGE, file.isImage())
            .setTransition(Transition.NONE)
            .startActivity();
      }
    }
  }

  @Override
  public void onActionClick(@NonNull String actionType, @NonNull ActionButton actionButton) {
    presenter.onActionClick(actionType, actionButton);
  }

  @Override
  public void onResendButtonClick(SendingMessageItem item) {
    final AlertDialog dialog = new AlertDialog.Builder(this)
        .setMessage(ResUtils.getString("ch.chat.resend.description"))
        .setPositiveButton(ResUtils.getString("ch.chat.retry_sending_message"), (dialog12, which) -> presenter.resend(item))
        .setNegativeButton(ResUtils.getString("ch.chat.resend.cancel"), null)
        .setNeutralButton(ResUtils.getString("ch.chat.delete"), (dialog1, which) -> presenter.removeFailedItem(item))
        .setCancelable(true)
        .create();

    dialog.setOnShowListener(args -> {
      int dark = ContextCompat.getColor(ChatActivity.this, R.color.ch_dark);
      int cobalt = ContextCompat.getColor(ChatActivity.this, R.color.ch_cobalt);

      dialog.getButton(AlertDialog.BUTTON_POSITIVE).setTextColor(cobalt);
      dialog.getButton(AlertDialog.BUTTON_NEGATIVE).setTextColor(dark);
      dialog.getButton(AlertDialog.BUTTON_NEUTRAL).setTextColor(dark);
    });
    dialog.show();
  }

  @Override
  public void onRedirectContentsClick(@Nullable String redirectUrl) {
    Executor.executeRedirectUrlAction(this, redirectUrl);
  }

  @Override
  public void onLocalImageClick(@Nullable Previewable previewThumb) {
    if (previewThumb != null) {
      IntentUtils.setNextActivity(this, PhotoAlbumActivity.class)
          .putExtra(Const.EXTRA_LOCAL_IMAGE_URL, previewThumb.getUrl())
          .putExtra(Const.EXTRA_LOCAL_IMAGE_MODE, true)
          .startActivityForResult(Const.REQUEST_PHOTO_ALBUM);
    }
  }

  @Override
  public void onAttachmentButtonClick() {
    IntentUtils.setNextActivity(this, PhotoPickerActivity.class).startActivityForResult(Const.PHOTO_REQUEST_CODE);
  }

  @Override
  public void onSendClick(String message) {
    presenter.sendText(message);
  }

  // profile bot listeners

  @Override
  public void onCountrySelectorClick(ProfileBotMessageItem item) {
    presenter.requestCountries(item);
  }

  @Override
  public void showCountryDialog(List<Country> countries, ProfileBotMessageItem item) {
    new CountryCodeDialog(this, countries, (position, countryCode) -> presenter.selectCountry(countryCode, item)).show();
  }

  @Override
  public void onProfileRequest(ProfileBotMessageItem item) {
    presenter.updateProfile(item);
  }

  @Override
  public void onProfileUpdateComplete() {
    inputView.setFocus();
  }

  @Override
  public void setInputDim(boolean enabled) {
    inputView.setInputDim(enabled);
  }

  // callbacks

  @Override
  protected void onActivityResult(int requestCode, int resultCode, Intent data) {
    super.onActivityResult(requestCode, resultCode, data);

    switch (requestCode) {
      case Const.PHOTO_REQUEST_CODE:
        if (resultCode == Const.PHOTO_RESULT_CODE) {
          presenter.uploadFiles(data.getParcelableArrayListExtra(Const.PHOTO_INTENT_KEY));
        }
        break;
    }
  }
}
