package com.zoyi.channel.plugin.android.action;

import android.support.annotation.Nullable;
import com.zoyi.channel.plugin.android.activity.chat.model.SendingMessageItem;
import com.zoyi.channel.plugin.android.enumerate.ActionType;
import com.zoyi.channel.plugin.android.global.Api;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.ActionButton;
import com.zoyi.channel.plugin.android.model.rest.Plugin;
import com.zoyi.channel.plugin.android.model.wrapper.MessageWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.MessagesWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.UserChatWrapper;
import com.zoyi.channel.plugin.android.network.HttpStatus;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.store2.PluginStore;
import com.zoyi.channel.plugin.android.util.RequestUtils;
import com.zoyi.okhttp3.MediaType;
import com.zoyi.okhttp3.MultipartBody;
import com.zoyi.okhttp3.RequestBody;
import com.zoyi.rx.functions.Action0;

import java.util.HashMap;
import java.util.Map;

public class ChatAction {

  public static void requestProfileBot(@Nullable String chatId) {
    Plugin plugin = PluginStore.get().pluginState.get();

    if (chatId != null && plugin != null && plugin.getId() != null) {
      Api.requestProfileBot(chatId, plugin.getId())
          .runBy(ActionType.REQUEST_PROFILE_BOT)
          .cancelBy(ActionType.SHUTDOWN)
          .run(new RestSubscriber<>());
    }
  }

  public static void createUserChat(@Nullable String pluginId, RestSubscriber<UserChatWrapper> subscriber) {
    if (pluginId != null) {
      Api.createUserChat(pluginId)
          .runBy(ActionType.CREATE_USER_CHAT)
          .cancelBy(ActionType.SHUTDOWN)
          .run(subscriber);
    }
  }

  public static void removeUserChat(@Nullable String chatId, Action0 onSuccess, Action0 onFail) {
    if (chatId != null) {
      Api.removeUserChat(chatId)
          .cancelBy(ActionType.SHUTDOWN)
          .run(new RestSubscriber<Void>() {
            @Override
            public void onError(RetrofitException error) {
              onFail.call();
            }

            @Override
            public void onSuccessWithNull() {
              onSuccess.call();
            }
          });
    } else {
      onFail.call();
    }
  }

  public static void fetchUserChat(@Nullable String chatId, RestSubscriber<UserChatWrapper> repo) {
    if (chatId != null) {
      Api.getUserChat(chatId)
          .runBy(ActionType.FETCH_USER_CHAT)
          .cancelBy(ActionType.SOCKET_DISCONNECTED, ActionType.CHAT_CLOSED, ActionType.SHUTDOWN)
          .run(repo);
    }
  }

  public static void fetchMessages(
      @Nullable String chatId,
      RestSubscriber<MessagesWrapper> repo
  ) {
    if (chatId != null) {
      Api.getMessages(chatId, Const.MESSAGE_ID_MAX, Const.MESSAGE_FETCH_LIMIT, Const.DESC)
          .runBy(ActionType.FETCH_MESSAGES)
          .cancelBy(ActionType.SOCKET_DISCONNECTED, ActionType.CHAT_CLOSED, ActionType.SHUTDOWN)
          .run(repo);
    }
  }

  public static void fetchBackwardMessages(
      @Nullable String chatId,
      String since,
      RestSubscriber<MessagesWrapper> repo
  ) {
    if (chatId != null) {
      Api.getMessages(chatId, since, Const.MESSAGE_FETCH_LIMIT, Const.DESC)
          .runBy(ActionType.FETCH_BACKWARD_MESSAGES)
          .cancelBy(ActionType.FETCH_MESSAGES, ActionType.CHAT_CLOSED, ActionType.SOCKET_DISCONNECTED, ActionType.SHUTDOWN)
          .run(repo);
    }
  }

  public static void sendTextMessage(@Nullable String chatId, SendingMessageItem item, RestSubscriber<MessageWrapper> subscriber) {
    if (chatId != null && item != null) {
      RequestBody body = RequestUtils.form()
          .set("message", item.getText())
          .set("requestId", item.getRequestId())
          .create();

      Api.sendMessage(chatId, body)
          .cancelBy(ActionType.SHUTDOWN)
          .run(subscriber);
    }
  }

  public static void uploadFile(@Nullable String chatId, SendingMessageItem item, RestSubscriber<MessageWrapper> subscriber) {
    if (chatId != null && item != null && item.getUri() != null) {
      MultipartBody.Part body = RequestUtils.makeMultipart(item.getUri());
      RequestBody timestamp = RequestBody.create(MediaType.parse("text/plane"), item.getRequestId());

      if (body == null) {
        subscriber.onError(new RetrofitException(HttpStatus.UNSUPPORTED_MEDIA_TYPE, "Not supported type"));
        return;
      }

      Api.uploadFile(chatId, body, timestamp)
          .cancelBy(ActionType.SHUTDOWN)
          .run(subscriber);
    }
  }

  public static void closeChat(@Nullable String chatId, SendingMessageItem item, RestSubscriber<MessageWrapper> subscriber) {
    ActionButton actionButton = item.getActionButton();

    if (chatId != null && actionButton != null) {
      Api.closeUserChat(chatId, actionButton.getMessage().getId(), item.getRequestId())
          .cancelBy(ActionType.SHUTDOWN)
          .run(subscriber);
    }
  }

  public static void reviewChat(@Nullable String chatId, String review, SendingMessageItem item,
      RestSubscriber<MessageWrapper> subscriber) {
    ActionButton actionButton = item.getActionButton();

    if (chatId != null && actionButton != null) {
      Api.reviewUserChat(chatId, actionButton.getMessage().getId(), review, item.getRequestId())
          .cancelBy(ActionType.SHUTDOWN)
          .run(subscriber);
    }
  }

  public static void selectForm(@Nullable String chatId, SendingMessageItem item, RestSubscriber<MessageWrapper> subscriber) {
    ActionButton actionButton = item.getActionButton();

    if (chatId != null && actionButton != null) {
      Map<String, String> submit = new HashMap<>();
      submit.put("id", actionButton.getMessage().getId());
      submit.put("key", actionButton.getKey());

      RequestBody body = RequestUtils.form()
          .set("message", item.getText())
          .set("requestId", item.getRequestId())
          .set("submit", submit)
          .create();

      Api.sendMessage(chatId, body)
          .cancelBy(ActionType.SHUTDOWN)
          .run(subscriber);
    }
  }

  public static void selectSupportBotStep(@Nullable String chatId, SendingMessageItem item, RestSubscriber<MessageWrapper> subscriber) {
    ActionButton actionButton = item.getActionButton();

    if (chatId != null && actionButton != null) {
      Api.submitButtonAction(chatId, actionButton.getKey(), actionButton.getMessage().getId(), item.getRequestId())
          .cancelBy(ActionType.SHUTDOWN)
          .run(subscriber);
    }
  }

  public static void read(@Nullable String chatId) {
    if (chatId != null) {
      Api.read(chatId).cancelBy(ActionType.SHUTDOWN).run(new RestSubscriber<>());
    }
  }
}
